package ru.yandex.qe.dispenser.ws.staff;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.base.Verify;

import ru.yandex.qe.dispenser.domain.staff.StaffGroupType;

public final class StaffGroupsQuery {

    private final String query;

    private StaffGroupsQuery(final String query) {
        this.query = query;
    }

    public Optional<String> getQuery() {
        return Optional.ofNullable(query);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static IdFilter idEquals(final long value) {
        return new IdEqualsFilter(value);
    }

    public static IdFilter idGreaterThan(final long value) {
        return new IdGreaterThanFilter(value);
    }

    public static ModifiedAtFilter modifiedAtLessThan(final Instant value) {
        return new ModifiedAtLessThanFilter(value);
    }

    public static ModifiedAtFilter modifiedAtBetween(final Instant greaterOrEqualsThan, final Instant lessThan) {
        return new ModifiedAtBetween(greaterOrEqualsThan, lessThan);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final StaffGroupsQuery that = (StaffGroupsQuery) o;
        return Objects.equals(query, that.query);
    }

    @Override
    public int hashCode() {
        return Objects.hash(query);
    }

    @Override
    public String toString() {
        return "StaffGroupsQuery{" +
                "query='" + query + '\'' +
                '}';
    }

    public static final class Builder {

        private boolean deleted;
        private IdFilter idFilter;
        private ModifiedAtFilter modifiedAtFilter;
        private String url;
        private EnumSet<StaffGroupType> types;

        private Builder() {
        }

        public Builder deleted(final boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder idFilter(final IdFilter idFilter) {
            this.idFilter = idFilter;
            return this;
        }

        public Builder modifiedAtFilter(final ModifiedAtFilter modifiedAtFilter) {
            this.modifiedAtFilter = modifiedAtFilter;
            return this;
        }

        public Builder url(final String url) {
            this.url = url;
            return this;
        }

        public Builder types(final StaffGroupType... types) {
            Verify.verify(types.length > 0, "At least one group type is required");
            Verify.verify(Arrays.stream(types).noneMatch(StaffGroupType.UNKNOWN::equals), "Filter by unknown group type is forbidden");
            this.types = EnumSet.copyOf(Arrays.asList(types));
            return this;
        }

        public StaffGroupsQuery build() {
            if (!deleted && idFilter == null && modifiedAtFilter == null && url == null) {
                return new StaffGroupsQuery(null);
            }
            final StringBuilder queryBuilder = new StringBuilder();
            if (deleted) {
                queryBuilder.append("is_deleted == True");
            }
            if (idFilter != null) {
                if (queryBuilder.length() > 0) {
                    queryBuilder.append(" and ");
                }
                queryBuilder.append(idFilter.toQuery());
            }
            if (modifiedAtFilter != null) {
                if (queryBuilder.length() > 0) {
                    queryBuilder.append(" and ");
                }
                queryBuilder.append(modifiedAtFilter.toQuery());
            }
            if (url != null) {
                if (queryBuilder.length() > 0) {
                    queryBuilder.append(" and ");
                }
                queryBuilder.append("url == '");
                queryBuilder.append(url);
                queryBuilder.append("'");
            }
            if (types != null && !types.isEmpty()) {
                if (queryBuilder.length() > 0) {
                    queryBuilder.append(" and ");
                }
                if (types.size() == 1) {
                    queryBuilder.append("type == '");
                    queryBuilder.append(types.iterator().next().getValue());
                    queryBuilder.append("'");
                } else {
                    queryBuilder.append("type in [");
                    queryBuilder.append(types.stream().map(type -> "'" + type.getValue() + "'").collect(Collectors.joining(", ")));
                    queryBuilder.append("]");
                }
            }
            return new StaffGroupsQuery(queryBuilder.toString());
        }
    }

    public interface IdFilter {

        String toQuery();

    }

    private static final class IdEqualsFilter implements IdFilter {

        private final long value;

        private IdEqualsFilter(final long value) {
            this.value = value;
        }

        @Override
        public String toQuery() {
            return "id == " + value;
        }

    }

    private static final class IdGreaterThanFilter implements IdFilter {

        private final long value;

        private IdGreaterThanFilter(final long value) {
            this.value = value;
        }

        @Override
        public String toQuery() {
            return "id > " + value;
        }

    }

    public interface ModifiedAtFilter {

        String toQuery();

    }

    private static final class ModifiedAtLessThanFilter implements ModifiedAtFilter {

        private final Instant value;

        private ModifiedAtLessThanFilter(final Instant value) {
            this.value = value;
        }

        @Override
        public String toQuery() {
            return "_meta.modified_at < '" + DateTimeFormatter.ISO_INSTANT.format(value) + "'";
        }

    }

    private static final class ModifiedAtBetween implements ModifiedAtFilter {

        private final Instant greaterOrEqualsThan;
        private final Instant lessThan;

        private ModifiedAtBetween(final Instant greaterOrEqualsThan, final Instant lessThan) {
            this.greaterOrEqualsThan = greaterOrEqualsThan;
            this.lessThan = lessThan;
        }

        @Override
        public String toQuery() {
            return "_meta.modified_at >= '" + DateTimeFormatter.ISO_INSTANT.format(greaterOrEqualsThan)
                    + "' and _meta.modified_at < '" + DateTimeFormatter.ISO_INSTANT.format(lessThan) + "'";
        }

    }

}
