package ru.yandex.qe.dispenser.ws.v2;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;

import com.opencsv.bean.AbstractBeanField;
import com.opencsv.bean.AbstractCsvConverter;
import com.opencsv.bean.CsvBindAndSplitByName;
import com.opencsv.bean.CsvBindByName;
import com.opencsv.bean.CsvCustomBindByName;
import com.opencsv.bean.HeaderColumnNameMappingStrategy;
import com.opencsv.bean.StatefulBeanToCsv;
import com.opencsv.bean.StatefulBeanToCsvBuilder;
import com.opencsv.bean.comparator.LiteralComparator;
import org.apache.commons.lang3.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.TestOnly;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.api.v1.DiQuotaKey;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.ServiceBase;
import ru.yandex.qe.dispenser.ws.ServiceEndpointUtils;
import ru.yandex.qe.dispenser.ws.param.EntitySpecFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaGetParams;
import ru.yandex.qe.dispenser.ws.param.ResourceFilterParam;

import static ru.yandex.qe.dispenser.domain.util.FunctionalUtils.withExceptionWrapping;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ENTITY_SPEC_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.LEAF_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.MEMBER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ORDER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.PROJECT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.RESOURCE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SEGMENT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SERVICE_PARAM;


@Path("/v2/quotas/export")
@org.springframework.stereotype.Service("quota-export")
@ParametersAreNonnullByDefault
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaExportService extends ServiceBase {
    private static final String CONTENT_DISPOSITION_TEMPLATE = "attachment; filename=%s";

    @GET
    @Path("/csv")
    @Produces(ServiceBase.TEXT_CSV_UTF_8)
    public Response exportToCsv(@QueryParam(RESOURCE_PARAM) final List<ResourceFilterParam> resourceParams,
                                @QueryParam(ENTITY_SPEC_PARAM) final List<EntitySpecFilterParam> entitySpecParams,
                                @QueryParam(SEGMENT_PARAM) final Set<String> segmentKeys,
                                @QueryParam(SERVICE_PARAM) final Set<Service> services,
                                @QueryParam(PROJECT_PARAM) final Set<Project> projects,
                                @QueryParam(MEMBER_PARAM) final Set<Person> members,
                                @QueryParam(LEAF_PARAM) @DefaultValue("false") final boolean leafProjects,
                                @QueryParam(ORDER_PARAM) final DiOrder order) {
        final QuotaGetParams quotaGetParams = new QuotaGetParams(resourceParams, entitySpecParams, segmentKeys, services, projects, members, leafProjects, order);
        ServiceEndpointUtils.memoizeServiceForQuotas(quotaGetParams);
        final StreamingOutput output = new StreamingOutput() {
            @Override
            public void write(final OutputStream output) throws IOException, WebApplicationException {
                final BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(output));

                final HeaderColumnNameMappingStrategy<QuotaCSVBean> mappingStrategy = new HeaderColumnNameMappingStrategy<>();
                mappingStrategy.setType(QuotaCSVBean.class);
                mappingStrategy.setColumnOrderOnWrite(QuotaCSVBean.getColumnOrderComparator());

                final StatefulBeanToCsv<QuotaCSVBean> beanToCsv = new StatefulBeanToCsvBuilder<QuotaCSVBean>(writer)
                        .withMappingStrategy(mappingStrategy)
                        .build();

                final Comparator<QuotaCSVBean> rowsOrderComparator = Comparator.comparing(QuotaCSVBean::getServiceKey)
                        .thenComparing(QuotaCSVBean::getResourceKey)
                        .thenComparing(QuotaCSVBean::getQuotaSpecKey)
                        .thenComparing(QuotaCSVBean::getProjectKey);

                quotaGetParams.getQuotaStream()
                        .map(QuotaCSVBean::fromQuota)
                        .sorted(rowsOrderComparator)
                        .forEach(withExceptionWrapping(beanToCsv::write));

                writer.close();
            }
        };

        return Response.ok(output)
                .header(HttpHeaders.CONTENT_DISPOSITION, String.format(CONTENT_DISPOSITION_TEMPLATE, "quotas.csv"))
                .build();
    }

    public static final class QuotaCSVBean {
        private static final String SERVICE_COLUMN = "SERVICE";
        private static final String RESOURCE_COLUMN = "RESOURCE";
        private static final String QUOTA_SPEC_COLUMN = "QUOTA SPEC";
        private static final String SEGMENTS_COLUMN = "SEGMENTS";
        private static final String PROJECT_COLUMN = "PROJECT";
        private static final String RESPONSIBLES_COLUMN = "PROJECT RESPONSIBLES";
        private static final String MAX_COLUMN = "MAX";
        private static final String ACTUAL_COLUMN = "ACTUAL";

        private static final String[] COLUMN_ORDER = {
                SERVICE_COLUMN, RESOURCE_COLUMN, QUOTA_SPEC_COLUMN, SEGMENTS_COLUMN,
                PROJECT_COLUMN, RESPONSIBLES_COLUMN, ACTUAL_COLUMN, MAX_COLUMN
        };

        @CsvBindByName(column = SERVICE_COLUMN)
        private String serviceKey;
        @CsvBindByName(column = RESOURCE_COLUMN)
        private String resourceKey;
        @CsvBindByName(column = QUOTA_SPEC_COLUMN)
        private String quotaSpecKey;
        @CsvBindAndSplitByName(column = SEGMENTS_COLUMN, elementType = String.class)
        private Set<String> segmentKeys;
        @CsvBindByName(column = PROJECT_COLUMN)
        private String projectKey;
        @CsvBindAndSplitByName(column = RESPONSIBLES_COLUMN, elementType = Person.class, converter = StringToPersonElementConverter.class)
        private Set<Person> projectResponsibles;
        @CsvCustomBindByName(column = MAX_COLUMN, converter = StringToDiAmountFieldConverter.class)
        private DiAmount max;
        @CsvCustomBindByName(column = ACTUAL_COLUMN, converter = StringToDiAmountFieldConverter.class)
        private DiAmount actual;

        public QuotaCSVBean() {
        }

        public QuotaCSVBean(@NotNull final String serviceKey, @NotNull final String resourceKey,
                            @NotNull final String quotaSpecKey, @NotNull final Set<String> segmentKeys,
                            @NotNull final String projectKey, @NotNull final Set<Person> projectResponsibles,
                            @NotNull final DiAmount max, @NotNull final DiAmount actual) {
            this.serviceKey = serviceKey;
            this.resourceKey = resourceKey;
            this.quotaSpecKey = quotaSpecKey;
            this.segmentKeys = segmentKeys;
            this.projectKey = projectKey;
            this.projectResponsibles = projectResponsibles;
            this.max = max;
            this.actual = actual;
        }

        @NotNull
        public static QuotaCSVBean fromQuota(final QuotaView quota) {
            final DiQuotaKey key = quota.getKey().toView();
            final Set<Person> responsibles = Hierarchy.get().getProjectReader().getLinkedResponsibles(quota.getProject());
            return new QuotaCSVBean(key.getServiceKey(), key.getResourceKey(), key.getQuotaSpecKey(), key.getSegmentKeys(),
                    key.getProjectKey(), responsibles, quota.getMaxAmount().humanize(), quota.getTotalActualAmount().humanize());
        }

        @NotNull
        public static Comparator<String> getColumnOrderComparator() {
            return new LiteralComparator<>(COLUMN_ORDER);
        }

        @TestOnly
        @NotNull
        public static String[] getColumnOrder() {
            return COLUMN_ORDER;
        }

        public String getServiceKey() {
            return serviceKey;
        }

        public String getResourceKey() {
            return resourceKey;
        }

        public String getQuotaSpecKey() {
            return quotaSpecKey;
        }

        public Set<String> getSegmentKeys() {
            return segmentKeys;
        }

        public String getProjectKey() {
            return projectKey;
        }

        public Set<Person> getProjectResponsibles() {
            return projectResponsibles;
        }

        public DiAmount getMax() {
            return max;
        }

        public DiAmount getActual() {
            return actual;
        }
    }

    public static class StringToPersonElementConverter extends AbstractCsvConverter {
        @Override
        public Object convertToRead(final String value) {
            if (StringUtils.isNotEmpty(value)) {
                return Hierarchy.get().getPersonReader().readPersonByLogin(value);
            } else {
                return null;
            }
        }
    }

    public static class StringToDiAmountFieldConverter<T> extends AbstractBeanField<T> {
        @Override
        protected Object convert(final String value) {
            return DiAmount.fromHumanizedString(value);
        }
    }
}
