import utils
import pandas as pd
import logging
import json
import psycopg2
import psycopg2.extras
import argparse

parser = argparse.ArgumentParser()

parser.add_argument("-d", "--dry-run", type=bool, default=False, nargs="?", const=True)
parser.add_argument("-p", "--production", type=bool, default=False, nargs="?", const=True)

args = parser.parse_args()

utils.configure_logging('3826/3826.log')

COMMENT = "DISPENSER-3826"

if args.production:
    conn = utils.common_prod_connection()
    campaign_order = 7
    big_order_id = 43
    filename = '3826/3826.csv'
else:
    conn = utils.common_testing_connection()
    campaign_order = 845
    big_order_id = 45
    filename = '3826/3826_test.csv'

table = pd.read_csv(filename, delimiter='|')
project_by_ticket = {}
for i, row in table.iterrows():
    project = row['PROJECT']
    for ticket in row['TICKET'].split(','):
        sticket = ticket.strip()
        if isinstance(project, str) and project != '':
            project_by_ticket[sticket] = project


dispenser = utils.Dispenser(args.production, 'common')
reqs = dispenser.get_requests(status=['CONFIRMED'], campaignOrder=campaign_order, service='nirvana')

cursor = conn.cursor(cursor_factory=psycopg2.extras.DictCursor)

for req in reqs.body:
    ticket = req['trackerIssueKey']
    nirvana_project = project_by_ticket.get(ticket)
    nirvana_changes = []
    not_allocated_nirvana_changes = []
    req_key = f"{req['id']} {ticket}"
    for change in req['changes']:
        if change['service']['key'] == 'nirvana' and \
            change['resource']['key'] == 'cpu' and \
            change['order']['id'] == big_order_id and \
            change['amount']['value'] > 0 and \
            change['amountAllocated']['value'] < change['amount']['value']:
            nirvana_changes.append(change)

            if change['amountReady']['value'] < change['amount']['value']:
                not_allocated_nirvana_changes.append(change)

    if not nirvana_changes:
        logging.info(f"Skipping request {req_key}")
        continue 

    if not_allocated_nirvana_changes:
        changes_to_update_ready = []
        for change in not_allocated_nirvana_changes:
            changes_to_update_ready.append({
                "serviceKey": change['service']['key'],
                "resourceKey": change['resource']['key'],
                "segmentKeys": change['segmentKeys'],
                "bigOrderId": change['order']['id'],
                "amountReady": {
                    "unit": change['amount']['unit'],
                    "value": change['amount']['value']
                }
            })

        update_ready_request = {
            "updates": [
                {
                    "requestId": req['id'],
                    "comment": COMMENT,
                    "changes": changes_to_update_ready
                }
            ]
        }

        logging.info(f"Body for {req_key}: {update_ready_request}")

        if not args.dry_run:
            logging.info(f"Updating ready amount for request {req_key}")
            r = dispenser.set_quota_state(update_ready_request)
            if r.status != 200:
                logging.error(f"Couldn't update ready amount for request {req_key}, status={r.status}: {r.body}")
                continue
    if nirvana_project:
        logging.info(f"Setting project '{nirvana_project}' for request {req_key}")
        if not args.dry_run:
            new_properties = json.dumps({"project": nirvana_project.strip()})
            new_data = json.dumps({
                "additionalProperties": {
                    "project": nirvana_project.strip()
                }
            })
            old_data = json.dumps({
                "additionalProperties": req.get('additionalProperties')
            })
            cursor.execute("UPDATE quota_request SET additional_properties = %s WHERE id = %s", (new_properties, req['id']))
            cursor.execute("INSERT INTO quota_request_history(quota_request_id, event_type, person_id, comment, updated, old_data, new_data)" + \
                " SELECT %s, 'FIELDS_UPDATE', p.id, %s, now(), %s, %s FROM person p where login = 'robot-dispenser'", (req['id'], COMMENT, old_data, new_data))
            conn.commit()
            logging.info(f"Successfully set project '{nirvana_project}' for request {req_key}")
    else: 
        logging.warning(f"Request {req_key} has no nirvana project!")

    if not args.dry_run and nirvana_project:
        r = dispenser.allocate_provider(req['id'], 'nirvana')
        if r.status != 200:
            logging.error(f"Couldn't allocate request {req_key}, status={r.status}: {r.body}")
    
        