import json
import logging
import datetime

import utils
import pandas as pd

PROD = False #fixme
DRY_RUN = True #fixme
TRACKER_TOKEN = 'YOUR_TOKEN' #todo fill
DISPENSER_TOKEN = 'YOUR_TOKEN' #todo fill
MONTH = ['', 'Январь', 'Февраль', 'Март', 'Апрель', 'Май', 'Июнь', 'Июль', 'Август', "Сентябрь", "Октябрь", "Ноябрь", "Декабрь"]
ST_HOST = "https://st.yandex-team.ru" if PROD else "https://st.test.yandex-team.ru"

EXCLUDED_PROVIDERS = {"logbroker", "logbroker_yt_proxy"} #todo fill

STORAGE_UNITS = {
    0: "B",
    1: "KiB",
    2: "MiB",
    3: "GiB",
    4: "TiB",
    5: "PiB"
}


def calc_storage_units(raw_amount):
    c = 0
    amount = raw_amount
    while amount % 1024 == 0:
        c += 1
        amount /= 1024
    return amount, STORAGE_UNITS[c]


def readable_amount_string(raw_amount, type):
    if type in ["MEMORY", "STORAGE"]:
        amount, unit = calc_storage_units(raw_amount)
        return f"{amount} {unit}"
    elif type in ["PROCESSOR"]:
        return f"{raw_amount / 1000} cores"
    elif type in ["BINARY_TRAFFIC"]:
        return f"{raw_amount / (2 ** 20)} MiBps"
    elif type in ["ENUMERABLE"]:
        return f"{raw_amount / 1000} items"
    elif type in ["TRAFFIC"]:
        return f"{raw_amount / 1000000} MBps"
    else:
        return str(raw_amount)


def prepare_request_item(ticket_key, providers):
    return {
        "ticketKey": ticket_key,
        "changes": [{"serviceKey": p} for p in providers]
    }


description = """Коллеги,

Как мы ((https://clubs.at.yandex-team.ru/capacity-planning/209 предупреждали ранее)), этот заказ списывается.

Будут списаны следующие ресурсы:
{body}
"""

if PROD:
    requests_file = 'DISPENSER-5298.csv'
    log_file = '5298.log'
else:
    requests_file = 'DISPENSER-5298-test.csv'
    log_file = '5298_test.log'

utils.configure_logging(log_file)
requests = pd.read_csv(requests_file)

requests_by_tickets = {}
order_dict = {}
tickets_to_confirm = set()

for i, row in requests.iterrows():
    ticket_key = row['ticket_key']
    if row['status'] != 'CONFIRMED':
        tickets_to_confirm.add(ticket_key)
    if ticket_key not in requests_by_tickets:
        requests_by_tickets[ticket_key] = {}

    req = requests_by_tickets[ticket_key]
    service_key = row['service_key']
    if service_key not in req:
        req[service_key] = []

    req[service_key].append(row)
    order_dict[int(row["order_id"])] = row["date"]

tracker = utils.Tracker(prod=PROD, override_token=TRACKER_TOKEN)
dispenser = utils.Dispenser(prod=PROD, cluster="common", override_token=DISPENSER_TOKEN)


tickets_to_close = set()

set_amount_body = {
    "updateFor": "BOTH",
    "updates": []
}

for ticket, providers in requests_by_tickets.items():
    has_logbroker = EXCLUDED_PROVIDERS.intersection(providers.keys())
    has_another_provider = providers.keys() - EXCLUDED_PROVIDERS
    if has_logbroker and not has_another_provider:
        logging.info(f"Skip request {ticket} because it has only logbroker resources")
        continue

    body = ""
    used_providers = []
    for provider, rows in providers.items():
        if provider in EXCLUDED_PROVIDERS:
            continue
        used_providers.append(provider)
        example = rows[0]
        body += f"### Ресурсы {example['service_name']}\n\n"
        resources = set()
        amount_by_order = {}
        for row in rows:
            resource_name = row['resource_name']
            resources.add(resource_name)
            raw_segments = row['segments']
            segments = ", ".join([x.strip('"') for x in raw_segments.strip("{}").split(",")])
            key = (int(row["order_id"]), segments)
            if key not in amount_by_order:
                amount_by_order[key] = {}
            resource_amounts = amount_by_order[key]
            resource_amounts[resource_name] = row
        body += "#|\n"
        body += "|| |" + " | ".join(resources) + " ||\n"
        for key in sorted(amount_by_order.keys(), key=lambda x: x[0]):
            order_id, segments = key
            resource_amounts = amount_by_order[key]
            iso_date = order_dict[order_id]
            date = datetime.date.fromisoformat(iso_date)
            body += f"|| {MONTH[date.month]} {date.year}{': ' + segments if segments else ''} | "
            values = []
            for rn in resources:
                if rn not in resource_amounts:
                    values.append(" ")
                    continue
                row = resource_amounts[rn]
                raw_amount = int(row["amount_available"])
                values.append(readable_amount_string(raw_amount, row["resource_type"]))
            body += " | ".join(values) + " ||\n"

        body += "|#\n\n"
    set_amount_body["updates"].append(prepare_request_item(ticket, used_providers))
    logging.info(f"Processing {ST_HOST}/{ticket}. Available providers: {used_providers}")
    prepared_description = description.format(body=body)
    logging.debug(prepared_description)
    if not has_logbroker:
        tickets_to_close.add(ticket)
    if not DRY_RUN:
        tracker.post_comment(ticket, prepared_description)
        pass

logging.info(f"Tickets to be confirmed: {tickets_to_confirm}")
if not DRY_RUN and tickets_to_confirm:
    result = dispenser.admin_change_status(tickets_to_confirm, "CONFIRMED", "DISPENSER-5298")
    logging.info(result)

logging.info(f"Tickets to be closed ({len(tickets_to_close)}): {tickets_to_close}")
logging.info(f"Request body: \n{json.dumps(set_amount_body, indent=4)}")
if not DRY_RUN:
    response = dispenser.set_quota_state_optional(set_amount_body, suppress_summon=True)
    logging.info(response)
