import json
import argparse
import requests
from utils import get_token
import logging
import sys

logging.basicConfig(format='%(asctime)s - [%(levelname)s] - %(message)s', level=logging.INFO)

TOKEN = get_token('robot-dispenser')

UNITS_BY_RESOURCE = {
    "cpu": "CORES",
    "ram": "GIBIBYTE",
    "ssd": "GIBIBYTE",
    "hdd": "GIBIBYTE",
}

parser = argparse.ArgumentParser()
parser.add_argument("--ram", type=int)
parser.add_argument("--ssd", type=int)
parser.add_argument("--cpu", type=int)
parser.add_argument("--hdd", type=int)
parser.add_argument("-d", "--destination", type=str, required=True) # destination project key
parser.add_argument("-s", "--source", type=str, required=True) # source project key
parser.add_argument("-t", "--ticket", required=True) # user ticket
parser.add_argument("-r", "--release", default=False, type=bool, nargs="?", const=True) # create and apply move request

args = parser.parse_args()

result = {
    "description": "",
    "comment": args.ticket,
    "calculations": "",
    "sourceProjectKey": args.source,
    "projectKey": args.destination,
    "type": "QUOTA_MOVE",
    "chartLinks": None,
    "chartLinksAbsenceExplanation": "",
    "summary": f"MDB quota move [{args.source} -> {args.destination}]",
    "changes": []
}

args_d = vars(args)

for resource in ["ram", "ssd", "hdd", "cpu"]:
    if args_d[resource] is not None:
        change = {
            "resourceKey": resource,
            "serviceKey": "mdb",
            "segmentKeys": [],
            "amount": {
                "value": args_d[resource],
                "unit": UNITS_BY_RESOURCE[resource]
            }
        }
        result["changes"].append(change)

print(json.dumps(result, indent=4))

if args.release:
    resp = requests.post("https://dispenser.yandex-team.ru/db/api/v1/quota-requests", data=json.dumps(result), headers={"Content-Type": "application/json", "Authorization": f"OAuth {TOKEN}"})
    if resp.status_code != 200:
        logging.error("Error while creating request [%s] %s", resp.status_code, resp.text)
        sys.exit(0)
    req = resp.json()["result"][0]
    logging.info("Created ticket %s [https://st.yandex-team.ru/%s]", req["id"], req["trackerIssueKey"])
    id = req['id']
    status_resp = requests.put(f"https://dispenser.yandex-team.ru/db/api/v1/quota-requests/{id}/status/APPLIED", headers={"Authorization": f"OAuth {TOKEN}"})
    logging.info("Requests update status - %s", status_resp.status_code)
    if status_resp.status_code != 200:
        logging.info("Error %s", status_resp.text)

