import json
import requests
import sys
import os
import typing
from argparse import ArgumentParser

import library.python.oauth as lpo


def get_token() -> str:
    client_id = 'd6f0f789cdd8407d8f542ac9338a35cd'
    client_secret = '4ad20bafb5ab4d6087986b3c88a39438'
    token = lpo.get_token(client_id, client_secret)
    return token


def get_url(env: str) -> str:
    if env == "prod":
        return "https://dispenser.yandex-team.ru/common/api/v1/base-resource-limits/human-readable"
    elif env == "test":
        return "https://dispenser.test.yandex-team.ru/common/api/v1/base-resource-limits/human-readable"
    else:
        return ""


def is_float(value: str) -> bool:
    try:
        float(value)
        return True
    except ValueError:
        return False


def to_amount_value(value: str):
    if value.isdigit():
        return int(value)
    else:
        return float(value)


def get_all(env: str, campaign_key: str, provider_key: str, output_file: str) -> int:
    if campaign_key == "":
        print("Empty campaign key")
        return os.EX_DATAERR
    if provider_key == "":
        print("Empty provider key")
        return os.EX_DATAERR
    if output_file == "":
        print("Empty output file name")
        return os.EX_DATAERR
    url: str = get_url(env)
    if url == "":
        print("Invalid environment")
        return os.EX_DATAERR
    print("Getting all limits for " + provider_key + " in " + campaign_key + ", saving to " + output_file + "...")
    token = get_token()
    request_result: requests.Response = requests.get(url + "/_report",
                                                     params={"campaignKey": campaign_key,
                                                             "providerKey": provider_key},
                                                     headers={"Authorization": "OAuth " + token,
                                                              "User-Agent": "base-limits-tool"})
    if request_result.status_code == 200:
        with open(output_file, mode="w", encoding="utf-8") as output_fd:
            output_fd.write(request_result.text)
        print("Limits were saved to file successfully")
        return os.EX_OK
    else:
        if "Content-Type" in request_result.headers and request_result.headers["Content-Type"]\
                .startswith("application/json"):
            error_json: typing.Any = request_result.json()
            if "errors" in error_json or "fieldErrors" in error_json or "details" in error_json:
                print("Failed to get base resource limits:")
                if "errors" in error_json:
                    print(json.dumps(error_json["errors"]))
                if "fieldErrors" in error_json:
                    print(json.dumps(error_json["fieldErrors"]))
                if "details" in error_json:
                    print(json.dumps(error_json["details"]))
                return os.EX_SOFTWARE
            else:
                print("Failed to get base resource limits:")
                print(request_result.text)
                return os.EX_SOFTWARE
        else:
            print("Failed to get base resource limits:")
            print(request_result.text)
            return os.EX_SOFTWARE


def put_all(env: str, campaign_key: str, provider_key: str, input_file: str) -> int:
    if campaign_key == "":
        print("Empty campaign key")
        return os.EX_DATAERR
    if provider_key == "":
        print("Empty provider key")
        return os.EX_DATAERR
    if input_file == "":
        print("Empty input file name")
        return os.EX_DATAERR
    url: str = get_url(env)
    if url == "":
        print("Invalid environment")
        return os.EX_DATAERR
    with open(input_file, mode="r", encoding="utf-8") as input_fd:
        json_body = json.load(input_fd)
        print("Setting all limits for " + provider_key + " in " + campaign_key + ", reading from " + input_file + "...")
        token = get_token()
        request_result: requests.Response = requests.put(url + "/_report",
                                                         json=json_body,
                                                         params={"campaignKey": campaign_key,
                                                                 "providerKey": provider_key},
                                                         headers={"Authorization": "OAuth " + token,
                                                                  "User-Agent": "base-limits-tool"})
        if request_result.status_code == 200:
            print("Limits were updated successfully")
            return os.EX_OK
        else:
            if "Content-Type" in request_result.headers and request_result.headers["Content-Type"]\
                    .startswith("application/json"):
                error_json: typing.Any = request_result.json()
                if "errors" in error_json or "fieldErrors" in error_json or "details" in error_json:
                    print("Failed to update base resource limits:")
                    if "errors" in error_json:
                        print(json.dumps(error_json["errors"]))
                    if "fieldErrors" in error_json:
                        print(json.dumps(error_json["fieldErrors"]))
                    if "details" in error_json:
                        print(json.dumps(error_json["details"]))
                    return os.EX_SOFTWARE
                else:
                    print("Failed to update base resource limits:")
                    print(request_result.text)
                    return os.EX_SOFTWARE
            else:
                print("Failed to update base resource limits:")
                print(request_result.text)
                return os.EX_SOFTWARE


def get_one(env: str, campaign_key: str, base_resource_key: str) -> int:
    if campaign_key == "":
        print("Empty campaign key")
        return os.EX_DATAERR
    if base_resource_key == "":
        print("Empty base resource key")
        return os.EX_DATAERR
    url: str = get_url(env)
    if url == "":
        print("Invalid environment")
        return os.EX_DATAERR
    token = get_token()
    request_result: requests.Response = requests.get(url + "/_reportOne",
                                                     params={"campaignKey": campaign_key,
                                                             "baseResourceKey": base_resource_key},
                                                     headers={"Authorization": "OAuth " + token,
                                                              "User-Agent": "base-limits-tool"})
    if request_result.status_code == 200:
        result_json: typing.Any = request_result.json()
        print(str(result_json["limit"]["value"]) + " " + result_json["limit"]["unit"])
        return os.EX_OK
    else:
        if "Content-Type" in request_result.headers and request_result.headers["Content-Type"]\
                .startswith("application/json"):
            error_json: typing.Any = request_result.json()
            if "errors" in error_json or "fieldErrors" in error_json or "details" in error_json:
                print("Failed to get base resource limit:")
                if "errors" in error_json:
                    print(json.dumps(error_json["errors"]))
                if "fieldErrors" in error_json:
                    print(json.dumps(error_json["fieldErrors"]))
                if "details" in error_json:
                    print(json.dumps(error_json["details"]))
                return os.EX_SOFTWARE
            else:
                print("Failed to get base resource limit:")
                print(request_result.text)
                return os.EX_SOFTWARE
        else:
            print("Failed to get base resource limit:")
            print(request_result.text)
            return os.EX_SOFTWARE


def put_one(env: str, campaign_key: str, base_resource_key: str, value: str, unit: str) -> int:
    if campaign_key == "":
        print("Empty campaign key")
        return os.EX_DATAERR
    if base_resource_key == "":
        print("Empty base resource key")
        return os.EX_DATAERR
    if value == "":
        print("Empty value")
        return os.EX_DATAERR
    if unit == "":
        print("Empty unit")
        return os.EX_DATAERR
    if not is_float(value):
        print("Value is not a number")
        return os.EX_DATAERR
    amount_value = to_amount_value(value)
    url: str = get_url(env)
    if url == "":
        print("Invalid environment")
        return os.EX_DATAERR
    payload = {"limit": {"value": amount_value, "unit": unit}}
    token = get_token()
    request_result: requests.Response = requests.put(url + "/_reportOne",
                                                     json=payload,
                                                     params={"campaignKey": campaign_key,
                                                             "baseResourceKey": base_resource_key},
                                                     headers={"Authorization": "OAuth " + token,
                                                              "User-Agent": "base-limits-tool"})
    if request_result.status_code == 200:
        result_json: typing.Any = request_result.json()
        print(str(result_json["limit"]["value"]) + " " + result_json["limit"]["unit"])
        return os.EX_OK
    else:
        if "Content-Type" in request_result.headers and request_result.headers["Content-Type"]\
                .startswith("application/json"):
            error_json: typing.Any = request_result.json()
            if "errors" in error_json or "fieldErrors" in error_json or "details" in error_json:
                print("Failed to get base resource limit:")
                if "errors" in error_json:
                    print(json.dumps(error_json["errors"]))
                if "fieldErrors" in error_json:
                    print(json.dumps(error_json["fieldErrors"]))
                if "details" in error_json:
                    print(json.dumps(error_json["details"]))
                return os.EX_SOFTWARE
            else:
                print("Failed to get base resource limit:")
                print(request_result.text)
                return os.EX_SOFTWARE
        else:
            print("Failed to get base resource limit:")
            print(request_result.text)
            return os.EX_SOFTWARE


def get_parser():
    args_parser = ArgumentParser()
    args_parser.add_argument("--get-all", nargs=3, metavar=('<campaign_key>', '<provider_key>', '<output_file>'),
                             required=False)
    args_parser.add_argument("--put-all", nargs=3, metavar=('<campaign_key>', '<provider_key>', '<output_file>'),
                             required=False)
    args_parser.add_argument("--get-one", nargs=2, metavar=('<campaign_key>', '<base_resource_key>'), required=False)
    args_parser.add_argument("--put-one", nargs=4,
                             metavar=('<campaign_key>', '<base_resource_key>', '<value>', '<unit>'), required=False)
    args_parser.add_argument("--test", action='store_true', help='use test server')
    return args_parser


if __name__ == '__main__':

    parser = get_parser()
    args = parser.parse_args()
    env = 'test' if args.test else 'prod'

    exit_code: int = os.EX_OK
    if args.get_all:
        exit_code = get_all(env, *args.get_all)
    elif args.put_all:
        exit_code = put_all(env, *args.put_all)
    elif args.get_one:
        exit_code = get_one(env, *args.get_one)
    elif args.put_one:
        exit_code = put_one(env, *args.put_one)
    else:
        parser.print_help()
        exit_code = os.EX_USAGE

    sys.exit(exit_code)
