package ru.yandex.dispenser.validation.client;

import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import io.netty.channel.ChannelOption;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollDatagramChannel;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import io.netty.resolver.AddressResolverGroup;
import io.netty.resolver.dns.DnsNameResolverBuilder;
import io.netty.resolver.dns.DnsServerAddressStreamProvider;
import io.netty.resolver.dns.DnsServerAddressStreamProviders;
import io.netty.resolver.dns.RoundRobinDnsAddressResolverGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.InvalidMediaTypeException;
import org.springframework.http.MediaType;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.ExchangeFilterFunctions;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;

import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest;
import ru.yandex.dispenser.validation.client.model.DispenserError;
import ru.yandex.dispenser.validation.client.model.DispenserErrorDescription;
import ru.yandex.dispenser.validation.client.model.ExpandQuotaChangeRequest;
import ru.yandex.dispenser.validation.client.model.QuotaRequest;
import ru.yandex.dispenser.validation.client.model.QuotaRequestList;
import ru.yandex.dispenser.validation.client.model.Status;

/**
 * Dispenser client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class DispenserClient {

    private static final Logger LOG = LoggerFactory.getLogger(DispenserClient.class);

    private static final int MAX_QUERIES_PER_RESOLVE = 3;
    private static final int MIN_TTL = 28800;
    private static final int MAX_TTL = Integer.MAX_VALUE;
    private static final AddressResolverGroup<?> RESOLVER_INSTANCE = prepareResolver();

    private final WebClient client;
    private final String userAgent;
    private final long timeoutMillis;
    private final String testingUrl;
    private final String productionUrl;
    private final ObjectReader errorReader;

    @SuppressWarnings("ParameterNumber")
    public DispenserClient(@Value("${dispenser.client.connectTimeoutMs}") int connectTimeoutMillis,
                           @Value("${dispenser.client.readTimeoutMs}") long readTimeoutMillis,
                           @Value("${dispenser.client.writeTimeoutMs}") long writeTimeoutMillis,
                           @Value("${dispenser.client.responseSizeLimitBytes}") long responseSizeLimitBytes,
                           @Value("${dispenser.client.timeoutMs}") long timeoutMillis,
                           @Value("${dispenser.client.userAgent}") String userAgent,
                           @Value("${dispenser.api.testing}") String testingUrl,
                           @Value("${dispenser.api.production}") String productionUrl,
                           ObjectMapper objectMapper) {
        this.userAgent = userAgent;
        this.timeoutMillis = timeoutMillis;
        this.client = createWebClient(connectTimeoutMillis, readTimeoutMillis, writeTimeoutMillis,
                responseSizeLimitBytes);
        this.testingUrl = testingUrl;
        this.productionUrl = productionUrl;
        this.errorReader = objectMapper.readerFor(DispenserErrorDescription.class);
    }

    public Mono<Response<QuotaRequest, DispenserError>> getQuotaRequest(
            long quotaRequestId, Set<ExpandQuotaChangeRequest> expand,
            Environment environment, String token) {
        try {
            String requestId = UUID.randomUUID().toString();
            UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl(environment))
                    .pathSegment("common", "api", "v1", "quota-requests", String.valueOf(quotaRequestId));
            if (!expand.isEmpty()) {
                builder.queryParam("expand", expand.stream().map(Enum::toString).collect(Collectors.toList()));
            }
            String uri = builder.toUriString();
            return client.get()
                    .uri(uri)
                    .header("X-Request-ID", requestId)
                    .header("Authorization", "OAuth " + token)
                    .accept(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.USER_AGENT, userAgent)
                    .exchangeToMono(response -> toResponse(response, QuotaRequest.class, requestId))
                    .timeout(Duration.ofMillis(timeoutMillis))
                    .onErrorResume(e -> toResponse(e, requestId));
        } catch (Throwable e) {
            return Mono.error(e);
        }
    }

    public Mono<Response<QuotaRequestList, DispenserError>> createQuotaRequest(
            CreateQuotaRequest quotaRequest, Long campaignId, Set<ExpandQuotaChangeRequest> expand,
            Environment environment, String token) {
        try {
            String requestId = UUID.randomUUID().toString();
            UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl(environment))
                    .pathSegment("common", "api", "v1", "quota-requests")
                    .queryParam("reqId", requestId);
            if (campaignId != null) {
                builder.queryParam("campaign", String.valueOf(campaignId));
            }
            if (!expand.isEmpty()) {
                builder.queryParam("expand", expand.stream().map(Enum::toString).collect(Collectors.toList()));
            }
            String uri = builder.toUriString();
            return client.post()
                    .uri(uri)
                    .header("X-Request-ID", requestId)
                    .header("Authorization", "OAuth " + token)
                    .accept(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.USER_AGENT, userAgent)
                    .contentType(MediaType.APPLICATION_JSON)
                    .bodyValue(quotaRequest)
                    .exchangeToMono(response -> toResponse(response, QuotaRequestList.class, requestId))
                    .timeout(Duration.ofMillis(timeoutMillis))
                    .onErrorResume(e -> toResponse(e, requestId));
        } catch (Throwable e) {
            return Mono.error(e);
        }
    }

    public Mono<Response<QuotaRequestList, DispenserError>> createQuotaRequests(
            List<CreateQuotaRequest> quotaRequests, Long campaignId, Set<ExpandQuotaChangeRequest> expand,
            Environment environment, String token) {
        try {
            String requestId = UUID.randomUUID().toString();
            UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl(environment))
                    .pathSegment("common", "api", "v1", "quota-requests", "batch")
                    .queryParam("reqId", requestId);
            if (campaignId != null) {
                builder.queryParam("campaign", String.valueOf(campaignId));
            }
            if (!expand.isEmpty()) {
                builder.queryParam("expand", expand.stream().map(Enum::toString).collect(Collectors.toList()));
            }
            String uri = builder.toUriString();
            return client.post()
                    .uri(uri)
                    .header("X-Request-ID", requestId)
                    .header("Authorization", "OAuth " + token)
                    .accept(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.USER_AGENT, userAgent)
                    .contentType(MediaType.APPLICATION_JSON)
                    .bodyValue(quotaRequests)
                    .exchangeToMono(response -> toResponse(response, QuotaRequestList.class, requestId))
                    .timeout(Duration.ofMillis(timeoutMillis))
                    .onErrorResume(e -> toResponse(e, requestId));
        } catch (Throwable e) {
            return Mono.error(e);
        }
    }

    public Mono<Response<QuotaRequest, DispenserError>> setQuotaRequestStatus(
            long quotaRequestId, Status status, Set<ExpandQuotaChangeRequest> expand,
            Environment environment, String token) {
        try {
            String requestId = UUID.randomUUID().toString();
            UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl(environment))
                    .pathSegment("common", "api", "v1", "quota-requests", String.valueOf(quotaRequestId),
                            "status", status.toString())
                    .queryParam("reqId", requestId);
            if (!expand.isEmpty()) {
                builder.queryParam("expand", expand.stream().map(Enum::toString).collect(Collectors.toList()));
            }
            String uri = builder.toUriString();
            return client.put()
                    .uri(uri)
                    .header("X-Request-ID", requestId)
                    .header("Authorization", "OAuth " + token)
                    .accept(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.USER_AGENT, userAgent)
                    .exchangeToMono(response -> toResponse(response, QuotaRequest.class, requestId))
                    .timeout(Duration.ofMillis(timeoutMillis))
                    .onErrorResume(e -> toResponse(e, requestId));
        } catch (Throwable e) {
            return Mono.error(e);
        }
    }

    private <T> Mono<Response<T, DispenserError>> toResponse(ClientResponse clientResponse,
                                                             Class<T> responseClass, String sentRequestId) {
        int statusCode = clientResponse.rawStatusCode();
        if (isSuccess(statusCode)) {
            String receivedRequestId = clientResponse.headers().asHttpHeaders().getFirst("X-Request-ID");
            return clientResponse.bodyToMono(responseClass).map(value -> Response.success(value, receivedRequestId,
                    sentRequestId));
        }
        return clientResponse.createException().flatMap(Mono::error);
    }

    private <T> Mono<Response<T, DispenserError>> toResponse(Throwable error, String sentRequestId) {
        if (error instanceof WebClientResponseException) {
            return Mono.just(toResponse((WebClientResponseException) error, sentRequestId));
        }
        if (error.getCause() instanceof WebClientResponseException) {
            return Mono.just(toResponse((WebClientResponseException) error.getCause(), sentRequestId));
        }
        if (Exceptions.isRetryExhausted(error) && error.getCause() != null) {
            return Mono.just(Response.failure(error.getCause(), sentRequestId));
        }
        return Mono.just(Response.failure(error, sentRequestId));
    }

    private boolean isSuccess(int statusCode) {
        return statusCode >= 200 && statusCode < 300;
    }

    private <T> Response<T, DispenserError> toResponse(WebClientResponseException ex, String sentRequestId) {
        int statusCode = ex.getRawStatusCode();
        String receivedRequestId = ex.getHeaders().getFirst("X-Request-ID");
        boolean isJsonBody = isJsonContent(ex);
        boolean isTextBody = isTextContent(ex);
        if (!isJsonBody && !isTextBody) {
            return Response.error(DispenserError.httpError(statusCode), receivedRequestId, sentRequestId);
        } else if (isTextBody) {
            return Response.error(DispenserError.httpTextError(statusCode, ex.getResponseBodyAsString()),
                    receivedRequestId, sentRequestId);
        } else {
            try {
                DispenserErrorDescription errorDescription = errorReader.readValue(ex.getResponseBodyAsString());
                return Response.error(DispenserError.httpExtendedError(statusCode, errorDescription),
                        receivedRequestId, sentRequestId);
            } catch (Exception e) {
                LOG.error("Failed to read Dispenser API error", e);
                return Response.error(DispenserError.httpTextError(statusCode,
                        ex.getResponseBodyAsString()), receivedRequestId, sentRequestId);
            }
        }
    }

    private boolean isJsonContent(WebClientResponseException ex) {
        try {
            MediaType contentType = ex.getHeaders().getContentType();
            if (contentType == null) {
                return false;
            }
            return MediaType.APPLICATION_JSON.includes(contentType);
        } catch (InvalidMediaTypeException e) {
            LOG.error("Unexpected Dispenser API response content type", e);
            return false;
        }
    }

    private boolean isTextContent(WebClientResponseException ex) {
        try {
            MediaType contentType = ex.getHeaders().getContentType();
            if (contentType == null) {
                return false;
            }
            return MediaType.TEXT_PLAIN.includes(contentType);
        } catch (InvalidMediaTypeException e) {
            LOG.error("Unexpected Dispenser API response content type", e);
            return false;
        }
    }

    private String baseUrl(Environment environment) {
        return switch (environment) {
            case TESTING -> testingUrl;
            case PRODUCTION -> productionUrl;
        };
    }

    private static WebClient createWebClient(int connectTimeoutMillis,
                                             long readTimeoutMillis,
                                             long writeTimeoutMillis,
                                             long responseSizeLimitBytes) {
        return WebClient.builder()
                .clientConnector(new ReactorClientHttpConnector(HttpClient.create()
                        .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
                        .resolver(RESOLVER_INSTANCE)
                        .doOnConnected(connection -> {
                            connection.addHandlerLast(new ReadTimeoutHandler(readTimeoutMillis,
                                    TimeUnit.MILLISECONDS));
                            connection.addHandlerLast(new WriteTimeoutHandler(writeTimeoutMillis,
                                    TimeUnit.MILLISECONDS));
                        })))
                .filter(ExchangeFilterFunctions.limitResponseSize(responseSizeLimitBytes))
                .codecs(configurer -> configurer
                        .defaultCodecs()
                        .maxInMemorySize(Math.toIntExact(responseSizeLimitBytes)))
                .build();
    }

    private static AddressResolverGroup<?> prepareResolver() {
        DnsNameResolverBuilder dnsResolverBuilder = new DnsNameResolverBuilder();
        Class<? extends DatagramChannel> channelType = Epoll.isAvailable()
                ? EpollDatagramChannel.class : NioDatagramChannel.class;
        DnsServerAddressStreamProvider nameServerProvider = DnsServerAddressStreamProviders.platformDefault();
        dnsResolverBuilder
                .channelType(channelType)
                .nameServerProvider(nameServerProvider)
                .maxQueriesPerResolve(MAX_QUERIES_PER_RESOLVE)
                .ttl(MIN_TTL, MAX_TTL);
        return new RoundRobinDnsAddressResolverGroup(dnsResolverBuilder);
    }

}
