package ru.yandex.dispenser.validation.client;

import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Response.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public interface Response<T, E> {

    static <R, S> Response<R, S> success(R value, String receivedRequestId, String sentRequestId) {
        return new Response<>() {
            @Override
            public <O> O match(Cases<R, O, S> cases) {
                return cases.success(value, receivedRequestId, sentRequestId);
            }

            @Override
            public <O> O match(TriFunction<R, String, String, O> onSuccess, BiFunction<Throwable, String, O> onFailure,
                               TriFunction<S, String, String, O> onError) {
                return onSuccess.apply(value, receivedRequestId, sentRequestId);
            }

            @Override
            public boolean isSuccess() {
                return true;
            }

            @Override
            public boolean isFailure() {
                return false;
            }

            @Override
            public boolean isError() {
                return false;
            }

            @Override
            public int hashCode() {
                return Objects.hash(value, receivedRequestId, sentRequestId);
            }

            @SuppressWarnings("unchecked")
            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof Response)) {
                    return false;
                }
                final Response<Object, Object> response = (Response<Object, Object>) obj;
                return response.match(new Cases<>() {
                    @Override
                    public Boolean success(Object otherValue, String otherReceivedRequestId,
                                           String otherSentRequestId) {
                        return Objects.equals(value, otherValue) &&
                                Objects.equals(receivedRequestId, otherReceivedRequestId) &&
                                Objects.equals(sentRequestId, otherSentRequestId);
                    }

                    @Override
                    public Boolean failure(Throwable otherError, String otherSentRequestId) {
                        return false;
                    }

                    @Override
                    public Boolean error(Object otherError, String otherReceivedRequestId,
                                         String otherSentRequestId) {
                        return false;
                    }
                });
            }

            @Override
            public String toString() {
                return "Response::success{"
                        + "value=" + value
                        + ", receivedRequestId='" + receivedRequestId + '\''
                        + ", sentRequestId='" + sentRequestId + '\''
                        + '}';
            }

        };
    }

    static <R, S> Response<R, S> failure(Throwable error, String sentRequestId) {
        return new Response<>() {
            @Override
            public <O> O match(Cases<R, O, S> cases) {
                return cases.failure(error, sentRequestId);
            }

            @Override
            public <O> O match(TriFunction<R, String, String, O> onSuccess, BiFunction<Throwable, String, O> onFailure,
                               TriFunction<S, String, String, O> onError) {
                return onFailure.apply(error, sentRequestId);
            }

            @Override
            public boolean isSuccess() {
                return false;
            }

            @Override
            public boolean isFailure() {
                return true;
            }

            @Override
            public boolean isError() {
                return false;
            }

            @Override
            public int hashCode() {
                return Objects.hash(error, sentRequestId);
            }

            @SuppressWarnings("unchecked")
            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof Response)) {
                    return false;
                }
                final Response<Object, Object> response = (Response<Object, Object>) obj;
                return response.match(new Cases<>() {
                    @Override
                    public Boolean success(Object otherValue, String otherReceivedRequestId,
                                           String otherSentRequestId) {
                        return false;
                    }

                    @Override
                    public Boolean failure(Throwable otherError, String otherSentRequestId) {
                        return Objects.equals(error, otherError) && Objects.equals(sentRequestId, otherSentRequestId);
                    }

                    @Override
                    public Boolean error(Object otherError, String otherReceivedRequestId,
                                         String otherSentRequestId) {
                        return false;
                    }
                });
            }

            @Override
            public String toString() {
                return "Response::failure{"
                        + "error=" + error
                        + ", sentRequestId='" + sentRequestId + '\''
                        + '}';
            }

        };
    }

    static <R, S> Response<R, S> error(S error, String receivedRequestId, String sentRequestId) {
        return new Response<>() {
            @Override
            public <O> O match(Cases<R, O, S> cases) {
                return cases.error(error, receivedRequestId, sentRequestId);
            }

            @Override
            public <O> O match(TriFunction<R, String, String, O> onSuccess, BiFunction<Throwable, String, O> onFailure,
                               TriFunction<S, String, String, O> onError) {
                return onError.apply(error, receivedRequestId, sentRequestId);
            }

            @Override
            public boolean isSuccess() {
                return false;
            }

            @Override
            public boolean isFailure() {
                return false;
            }

            @Override
            public boolean isError() {
                return true;
            }

            @Override
            public int hashCode() {
                return Objects.hash(error, receivedRequestId, sentRequestId);
            }

            @SuppressWarnings("unchecked")
            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof Response)) {
                    return false;
                }
                final Response<Object, Object> response = (Response<Object, Object>) obj;
                return response.match(new Cases<>() {
                    @Override
                    public Boolean success(Object otherValue, String otherReceivedRequestId,
                                           String otherSentRequestId) {
                        return false;
                    }

                    @Override
                    public Boolean failure(Throwable otherError, String otherSentRequestId) {
                        return false;
                    }

                    @Override
                    public Boolean error(Object otherError, String otherReceivedRequestId,
                                         String otherSentRequestId) {
                        return Objects.equals(error, otherError)
                                && Objects.equals(receivedRequestId, otherReceivedRequestId)
                                && Objects.equals(sentRequestId, otherSentRequestId);
                    }
                });
            }

            @Override
            public String toString() {
                return "Response::error{"
                        + "error=" + error
                        + ", receivedRequestId='" + receivedRequestId + '\''
                        + ", sentRequestId='" + sentRequestId + '\''
                        + '}';
            }

        };
    }

    <R> R match(Cases<T, R, E> cases);

    <R> R match(TriFunction<T, String, String, R> onSuccess, BiFunction<Throwable, String, R> onFailure,
                TriFunction<E, String, String, R> onError);

    boolean isSuccess();

    boolean isFailure();

    boolean isError();

    interface Cases<I, O, E> {

        O success(I result, String receivedRequestId, String sentRequestId);

        O failure(Throwable error, String sentRequestId);

        O error(E error, String receivedRequestId, String sentRequestId);

    }

    @FunctionalInterface
    interface TriFunction<T, U, V, R> {
        R apply(T t, U u, V v);

        default <W> TriFunction<T, U, V, W> andThen(Function<? super R, ? extends W> after) {
            Objects.requireNonNull(after);
            return (T t, U u, V v) -> after.apply(apply(t, u, v));
        }

    }

}
