package ru.yandex.dispenser.validation.client.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Create change.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class CreateChange {

    private final Long orderId;
    private final String serviceKey;
    private final String resourceKey;
    private final Set<String> segmentKeys;
    private final Amount amount;

    @JsonCreator
    public CreateChange(@JsonProperty("orderId") Long orderId,
                        @JsonProperty("serviceKey") String serviceKey,
                        @JsonProperty("resourceKey") String resourceKey,
                        @JsonProperty("segmentKeys") Set<String> segmentKeys,
                        @JsonProperty("amount") Amount amount) {
        this.orderId = orderId;
        this.serviceKey = serviceKey;
        this.resourceKey = resourceKey;
        this.segmentKeys = segmentKeys;
        this.amount = amount;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(CreateChange createChange) {
        return new Builder(createChange);
    }

    public Long getOrderId() {
        return orderId;
    }

    public String getServiceKey() {
        return serviceKey;
    }

    public String getResourceKey() {
        return resourceKey;
    }

    public Set<String> getSegmentKeys() {
        return segmentKeys;
    }

    public Amount getAmount() {
        return amount;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreateChange that = (CreateChange) o;
        return Objects.equals(orderId, that.orderId) &&
                Objects.equals(serviceKey, that.serviceKey) &&
                Objects.equals(resourceKey, that.resourceKey) &&
                Objects.equals(segmentKeys, that.segmentKeys) &&
                Objects.equals(amount, that.amount);
    }

    @Override
    public int hashCode() {
        return Objects.hash(orderId, serviceKey, resourceKey, segmentKeys, amount);
    }

    @Override
    public String toString() {
        return "CreateChange{" +
                "orderId=" + orderId +
                ", serviceKey='" + serviceKey + '\'' +
                ", resourceKey='" + resourceKey + '\'' +
                ", segmentKeys=" + segmentKeys +
                ", amount=" + amount +
                '}';
    }

    public static final class Builder {

        private final Set<String> segmentKeys = new HashSet<>();

        private Long orderId;
        private String serviceKey;
        private String resourceKey;
        private Amount amount;

        private Builder() {
        }

        private Builder(CreateChange createChange) {
            this.orderId = createChange.orderId;
            this.serviceKey = createChange.serviceKey;
            this.resourceKey = createChange.resourceKey;
            this.amount = createChange.amount;
            this.segmentKeys.addAll(createChange.segmentKeys);
        }

        public Builder orderId(Long orderId) {
            this.orderId = orderId;
            return this;
        }

        public Builder serviceKey(String serviceKey) {
            this.serviceKey = serviceKey;
            return this;
        }

        public Builder resourceKey(String resourceKey) {
            this.resourceKey = resourceKey;
            return this;
        }

        public Builder amount(Amount amount) {
            this.amount = amount;
            return this;
        }

        public Builder addSegmentKey(String segmentKey) {
            this.segmentKeys.add(segmentKey);
            return this;
        }

        public Builder addSegmentKeys(Collection<? extends String> segmentKeys) {
            this.segmentKeys.addAll(segmentKeys);
            return this;
        }

        public Builder addSegmentKeys(String... segmentKeys) {
            this.segmentKeys.addAll(Arrays.asList(segmentKeys));
            return this;
        }

        public CreateChange build() {
            return new CreateChange(orderId, serviceKey, resourceKey, segmentKeys, amount);
        }

    }

}
