package ru.yandex.dispenser.validation.providers;

import java.util.List;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import ru.yandex.dispenser.validation.client.CampaignType;
import ru.yandex.dispenser.validation.client.Environment;
import ru.yandex.dispenser.validation.client.OrdersSubset;
import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest;
import ru.yandex.dispenser.validation.client.model.Reason;
import ru.yandex.dispenser.validation.client.model.Type;

/**
 * Validator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Configuration
@ConfigurationProperties
public class CommonSettings {

    private final String serviceTesting;
    private final String serviceProduction;
    private final Environment environment;
    private final AggregatedCampaignProperties aggregatedCampaignProperties;
    private final DraftCampaignProperties draftCampaignProperties;
    private final AggregatedOrdersProperties aggregatedOrdersProperties;
    private final DraftOrdersProperties draftOrdersProperties;
    private final ReducedAggregatedOrdersProperties reducedAggregatedOrdersProperties;
    private final ReducedDraftOrdersProperties reducedDraftOrdersProperties;

    @SuppressWarnings("ParameterNumber")
    public CommonSettings(@Value("${settings.service.testing}") String serviceTesting,
                          @Value("${settings.service.production}") String serviceProduction,
                          @Value("${validation.environment}") String environment,
                          AggregatedCampaignProperties aggregatedCampaignProperties,
                          DraftCampaignProperties draftCampaignProperties,
                          AggregatedOrdersProperties aggregatedOrdersProperties,
                          DraftOrdersProperties draftOrdersProperties,
                          ReducedAggregatedOrdersProperties reducedAggregatedOrdersProperties,
                          ReducedDraftOrdersProperties reducedDraftOrdersProperties) {
        this.serviceTesting = serviceTesting;
        this.serviceProduction = serviceProduction;
        this.environment = Environment.valueOf(environment);
        this.aggregatedCampaignProperties = aggregatedCampaignProperties;
        this.draftCampaignProperties = draftCampaignProperties;
        this.aggregatedOrdersProperties = aggregatedOrdersProperties;
        this.draftOrdersProperties = draftOrdersProperties;
        this.reducedAggregatedOrdersProperties = reducedAggregatedOrdersProperties;
        this.reducedDraftOrdersProperties = reducedDraftOrdersProperties;

    }

    public String service() {
        return switch (environment) {
            case TESTING -> serviceTesting;
            case PRODUCTION -> serviceProduction;
        };
    }

    public List<Long> orders(CampaignType campaignType, OrdersSubset ordersSubset) {
        return switch (campaignType) {
            case AGGREGATED -> aggregatedOrders(ordersSubset);
            case DRAFT -> draftOrders(ordersSubset);
        };
    }

    public long campaignId(CampaignType campaignType) {
        return switch (campaignType) {
            case AGGREGATED -> aggregatedCampaignId();
            case DRAFT -> draftCampaignId();
        };
    }

    public String textPlaceholder(String provider) {
        return "Test for " + provider + " base resource mappings";
    }

    public CreateQuotaRequest.Builder baseQuotaRequest(String provider) {
        return CreateQuotaRequest.builder()
                .description(textPlaceholder(provider))
                .comment(textPlaceholder(provider))
                .calculations(textPlaceholder(provider))
                .chartLinksAbsenceExplanation(textPlaceholder(provider))
                .resourcePreorderReasonType(Reason.GROWTH)
                .summary(textPlaceholder(provider))
                .projectKey(service())
                .type(Type.RESOURCE_PREORDER);
    }

    private List<Long> draftOrders(OrdersSubset ordersSubset) {
        return switch (ordersSubset) {
            case FULL -> fullDraftOrders();
            case REDUCED -> reducedDraftOrders();
        };
    }

    private List<Long> aggregatedOrders(OrdersSubset ordersSubset) {
        return switch (ordersSubset) {
            case FULL -> fullAggregatedOrders();
            case REDUCED -> reducedAggregatedOrders();
        };
    }

    private List<Long> fullDraftOrders() {
        return switch (environment) {
            case TESTING -> draftOrdersProperties.getTesting();
            case PRODUCTION -> draftOrdersProperties.getProduction();
        };
    }

    private List<Long> fullAggregatedOrders() {
        return switch (environment) {
            case TESTING -> aggregatedOrdersProperties.getTesting();
            case PRODUCTION -> aggregatedOrdersProperties.getProduction();
        };
    }

    private List<Long> reducedDraftOrders() {
        return switch (environment) {
            case TESTING -> reducedDraftOrdersProperties.getTesting();
            case PRODUCTION -> reducedDraftOrdersProperties.getProduction();
        };
    }

    private List<Long> reducedAggregatedOrders() {
        return switch (environment) {
            case TESTING -> reducedAggregatedOrdersProperties.getTesting();
            case PRODUCTION -> reducedAggregatedOrdersProperties.getProduction();
        };
    }

    private long draftCampaignId() {
        return switch (environment) {
            case TESTING -> draftCampaignProperties.getTesting();
            case PRODUCTION -> draftCampaignProperties.getProduction();
        };
    }

    private long aggregatedCampaignId() {
        return switch (environment) {
            case TESTING -> aggregatedCampaignProperties.getTesting();
            case PRODUCTION -> aggregatedCampaignProperties.getProduction();
        };
    }

}
