package ru.yandex.dispenser.validation.providers.logbroker;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.dispenser.validation.client.CampaignType;
import ru.yandex.dispenser.validation.client.Environment;
import ru.yandex.dispenser.validation.client.ErrorCollection;
import ru.yandex.dispenser.validation.client.OrdersSubset;
import ru.yandex.dispenser.validation.client.Result;
import ru.yandex.dispenser.validation.client.model.Amount;
import ru.yandex.dispenser.validation.client.model.CreateChange;
import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest;
import ru.yandex.dispenser.validation.client.model.QuotaRequest;
import ru.yandex.dispenser.validation.client.model.Units;
import ru.yandex.dispenser.validation.providers.CommonSettings;
import ru.yandex.dispenser.validation.providers.ProviderValidator;
import ru.yandex.dispenser.validation.providers.RequestedAmount;
import ru.yandex.dispenser.validation.providers.RequestedResource;
import ru.yandex.dispenser.validation.providers.SegmentedResourceKey;

/**
 * Logbroker validator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class LogbrokerValidator implements ProviderValidator {

    private static final Logger LOG = LoggerFactory.getLogger(LogbrokerValidator.class);

    private static final String PROVIDER_KEY = "logbroker";

    private final CommonSettings commonSettings;
    private final List<RequestedResource> resources;
    private final Set<String> segmentKeys;
    private final Set<String> reducedOrderSubsetSegmentKeys;

    public LogbrokerValidator(CommonSettings commonSettings) {
        this.commonSettings = commonSettings;
        this.resources = List.of(new RequestedResource("data_flow_binary",
                Units.BinaryBytesPerSecond.MEBIBYTES_PER_SECOND));
        this.segmentKeys = Set.of("lbkx", "logbroker_SAS", "logbroker_VLA", "logbroker_KLG", "logbroker-prestable_KLG",
                "logbroker-yc", "logbroker-yc-preprod");
        this.reducedOrderSubsetSegmentKeys = Set.of("logbroker_KLG", "logbroker-prestable_KLG");
    }

    @Override
    public CreateQuotaRequest prepareTestRequest(Environment environment, CampaignType campaignType) {
        CreateQuotaRequest.Builder builder = commonSettings.baseQuotaRequest(providerName());
        List<Long> orders = commonSettings.orders(campaignType, OrdersSubset.FULL);
        List<Long> ordersReducedSubset = commonSettings.orders(campaignType, OrdersSubset.REDUCED);
        long currentValue = 1L;
        for (RequestedResource resource : resources) {
            for (String segmentKey : segmentKeys) {
                List<Long> selectedOrders = reducedOrderSubsetSegmentKeys.contains(segmentKey)
                        ? ordersReducedSubset : orders;
                for (Long orderId : selectedOrders) {
                    long value = currentValue++;
                    builder.addChange(CreateChange.builder()
                            .serviceKey(PROVIDER_KEY)
                            .orderId(orderId)
                            .resourceKey(resource.getKey())
                            .addSegmentKey(segmentKey)
                            .amount(Amount.builder()
                                    .value(value)
                                    .unit(resource.getUnit())
                                    .build())
                            .build());
                }
            }
        }
        return builder.build();
    }

    @Override
    public Result<String> validateTestRequest(QuotaRequest request, Environment environment,
                                              CampaignType campaignType) {
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> requestedResources = new HashMap<>();
        request.getChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getService().getKey(),
                    change.getResource().getKey(), change.getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getAmount().getValue(),
                    Units.UNITS.get(change.getAmount().getUnit()));
            requestedResources.computeIfAbsent(change.getOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> baseResources = new HashMap<>();
        request.getBaseResourceChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getBaseResource().getType()
                    .getProvider().getKey(), change.getBaseResource().getType().getKey(),
                    change.getBaseResource().getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getTotalAmount().getValue(),
                    Units.UNITS.get(change.getTotalAmount().getUnit()));
            baseResources.computeIfAbsent(change.getBigOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> expectedBaseResources = new HashMap<>();
        for (Map.Entry<Long, Map<SegmentedResourceKey, RequestedAmount>> entry : requestedResources.entrySet()) {
            Long orderId = entry.getKey();
            Map<SegmentedResourceKey, RequestedAmount> resourcesMap = entry.getValue();
            for (String segmentKey : segmentKeys) {
                RequestedAmount requestedDataflow = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                        "data_flow_binary", Set.of(segmentKey)));
                if (requestedDataflow != null) {
                    expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                            .put(new SegmentedResourceKey(PROVIDER_KEY, "logbroker:dataflow",
                                    Set.of(segmentKey)), requestedDataflow);
                }
            }
        }
        LOG.info("Expected {} base resources: {}", providerName(), expectedBaseResources);
        LOG.info("Actual {} base resources: {}", providerName(), baseResources);
        boolean match = baseResources.equals(expectedBaseResources);
        if (match) {
            return Result.success(providerName());
        } else {
            return Result.failure(ErrorCollection.builder().addError("Base resource mismatch for " + providerName())
                    .build());
        }
    }

    @Override
    public String providerName() {
        return "Logbroker";
    }
}
