package ru.yandex.dispenser.validation.providers.mdb;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.dispenser.validation.client.CampaignType;
import ru.yandex.dispenser.validation.client.Environment;
import ru.yandex.dispenser.validation.client.ErrorCollection;
import ru.yandex.dispenser.validation.client.OrdersSubset;
import ru.yandex.dispenser.validation.client.Result;
import ru.yandex.dispenser.validation.client.model.Amount;
import ru.yandex.dispenser.validation.client.model.CreateChange;
import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest;
import ru.yandex.dispenser.validation.client.model.QuotaRequest;
import ru.yandex.dispenser.validation.client.model.Units;
import ru.yandex.dispenser.validation.providers.CommonSettings;
import ru.yandex.dispenser.validation.providers.ProviderValidator;
import ru.yandex.dispenser.validation.providers.RequestedAmount;
import ru.yandex.dispenser.validation.providers.RequestedResource;
import ru.yandex.dispenser.validation.providers.SegmentedResourceKey;

/**
 * MDB validator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class MdbValidator implements ProviderValidator {

    private static final Logger LOG = LoggerFactory.getLogger(MdbValidator.class);

    private static final String PROVIDER_KEY = "dbaas";

    private final CommonSettings commonSettings;
    private final List<RequestedResource> resources;

    public MdbValidator(CommonSettings commonSettings) {
        this.commonSettings = commonSettings;
        this.resources = List.of(new RequestedResource("cpu_all", Units.Cores.CORES),
                new RequestedResource("ram_all", Units.Bytes.GIBIBYTES),
                new RequestedResource("ssd_all", Units.Bytes.GIBIBYTES),
                new RequestedResource("hdd_all", Units.Bytes.GIBIBYTES));
    }

    @Override
    public CreateQuotaRequest prepareTestRequest(Environment environment, CampaignType campaignType) {
        CreateQuotaRequest.Builder builder = commonSettings.baseQuotaRequest(providerName());
        List<Long> orders = commonSettings.orders(campaignType, OrdersSubset.FULL);
        long currentValue = 1L;
        for (RequestedResource resource : resources) {
            for (Long orderId : orders) {
                long value = currentValue++;
                builder.addChange(CreateChange.builder()
                        .serviceKey(PROVIDER_KEY)
                        .orderId(orderId)
                        .resourceKey(resource.getKey())
                        .amount(Amount.builder()
                                .value(value)
                                .unit(resource.getUnit())
                                .build())
                        .build());
            }
        }
        return builder.build();
    }

    @Override
    public Result<String> validateTestRequest(QuotaRequest request, Environment environment,
                                              CampaignType campaignType) {
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> requestedResources = new HashMap<>();
        request.getChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getService().getKey(),
                    change.getResource().getKey(), change.getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getAmount().getValue(),
                    Units.UNITS.get(change.getAmount().getUnit()));
            requestedResources.computeIfAbsent(change.getOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> baseResources = new HashMap<>();
        request.getBaseResourceChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getBaseResource().getType()
                    .getProvider().getKey(), change.getBaseResource().getType().getKey(),
                    change.getBaseResource().getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getTotalAmount().getValue(),
                    Units.UNITS.get(change.getTotalAmount().getUnit()));
            baseResources.computeIfAbsent(change.getBigOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> expectedBaseResources = new HashMap<>();
        for (Map.Entry<Long, Map<SegmentedResourceKey, RequestedAmount>> entry : requestedResources.entrySet()) {
            Long orderId = entry.getKey();
            Map<SegmentedResourceKey, RequestedAmount> orderResources = entry.getValue();
            RequestedAmount requestedHdd = orderResources.get(new SegmentedResourceKey(PROVIDER_KEY,
                    "hdd_all", Set.of()));
            if (!requestedHdd.getUnit().equals(Units.Bytes.BYTES)) {
                LOG.error("Unexpected HDD unit: {}", requestedHdd);
                return Result.failure(ErrorCollection.builder().addError("Unexpected HDD unit").build());
            }
            RequestedAmount baseHdd = new RequestedAmount(toGibibytes(requestedHdd.getAmount()),
                    Units.Gibibytes.GIBIBYTES);
            RequestedAmount requestedSsd = orderResources.get(new SegmentedResourceKey(PROVIDER_KEY,
                    "ssd_all", Set.of()));
            if (!requestedSsd.getUnit().equals(Units.Bytes.BYTES)) {
                LOG.error("Unexpected SSD unit: {}", requestedSsd);
                return Result.failure(ErrorCollection.builder().addError("Unexpected SSD unit").build());
            }
            RequestedAmount baseSsd = new RequestedAmount(toGibibytes(requestedSsd.getAmount()),
                    Units.Gibibytes.GIBIBYTES);
            RequestedAmount requestedRam = orderResources.get(new SegmentedResourceKey(PROVIDER_KEY,
                    "ram_all", Set.of()));
            if (!requestedRam.getUnit().equals(Units.Bytes.BYTES)) {
                LOG.error("Unexpected RAM unit: {}", requestedRam);
                return Result.failure(ErrorCollection.builder().addError("Unexpected RAM unit").build());
            }
            RequestedAmount baseRam = new RequestedAmount(toGibibytes(requestedRam.getAmount()),
                    Units.Gibibytes.GIBIBYTES);
            RequestedAmount requestedCpu = orderResources.get(new SegmentedResourceKey(PROVIDER_KEY,
                    "cpu_all", Set.of()));
            if (!requestedCpu.getUnit().equals(Units.Cores.PERMILLE_CORES)) {
                LOG.error("Unexpected CPU unit: {}", requestedCpu);
                return Result.failure(ErrorCollection.builder().addError("Unexpected CPU unit").build());
            }
            RequestedAmount baseCpu = new RequestedAmount(requestedCpu.getAmount(),
                    Units.Cores.PERMILLE_CORES);
            expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                    .put(new SegmentedResourceKey(PROVIDER_KEY, "dbaas:hdd_all",
                            Set.of()), baseHdd);
            expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                    .put(new SegmentedResourceKey(PROVIDER_KEY, "dbaas:ssd_all",
                            Set.of()), baseSsd);
            expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                    .put(new SegmentedResourceKey(PROVIDER_KEY, "dbaas:ram_all",
                            Set.of()), baseRam);
            expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                    .put(new SegmentedResourceKey(PROVIDER_KEY, "dbaas:cpu_all",
                            Set.of()), baseCpu);
        }
        LOG.info("Expected {} base resources: {}", providerName(), expectedBaseResources);
        LOG.info("Actual {} base resources: {}", providerName(), baseResources);
        boolean match = baseResources.equals(expectedBaseResources);
        if (match) {
            return Result.success(providerName());
        } else {
            return Result.failure(ErrorCollection.builder().addError("Base resource mismatch for " + providerName())
                    .build());
        }
    }

    @Override
    public String providerName() {
        return "MDB";
    }

    private long toGibibytes(long bytes) {
        return bytes / (1024L * 1024L * 1024L);
    }

}
