package ru.yandex.dispenser.validation.providers.mds

import mu.KotlinLogging
import org.springframework.stereotype.Component
import ru.yandex.dispenser.validation.client.CampaignType
import ru.yandex.dispenser.validation.client.Environment
import ru.yandex.dispenser.validation.client.ErrorCollection
import ru.yandex.dispenser.validation.client.OrdersSubset
import ru.yandex.dispenser.validation.client.Result
import ru.yandex.dispenser.validation.client.model.Amount
import ru.yandex.dispenser.validation.client.model.CreateChange
import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest
import ru.yandex.dispenser.validation.client.model.QuotaRequest
import ru.yandex.dispenser.validation.client.model.Units
import ru.yandex.dispenser.validation.providers.CommonSettings
import ru.yandex.dispenser.validation.providers.ProviderValidator
import ru.yandex.dispenser.validation.providers.RequestedAmount
import ru.yandex.dispenser.validation.providers.RequestedResource
import ru.yandex.dispenser.validation.providers.SegmentedResourceKey

private val logger = KotlinLogging.logger {}

@Component
class S3Validator(
    private val commonSettings: CommonSettings
): ProviderValidator {

    private val s3ProviderKey = "s3"
    private val nirvanaProviderKey = "nirvana"
    private val sandboxProviderKey = "sandbox"
    private val mdbProviderKey = "dbaas"
    private val s3Resources = setOf(
        RequestedResource("x2_hdd", Units.Bytes.TEBIBYTES),
        RequestedResource("x3_hdd", Units.Bytes.TEBIBYTES)
    )
    private val nirvanaResources = setOf(
        RequestedResource("s3_hdd_x2", Units.Bytes.TEBIBYTES)
    )
    private val sandboxResources = setOf(
        RequestedResource("s3_hdd_x2", Units.Bytes.TEBIBYTES)
    )
    private val mdbResources = setOf(
        RequestedResource("s3_hdd_x3", Units.Bytes.TEBIBYTES)
    )

    override fun prepareTestRequest(environment: Environment, campaignType: CampaignType): CreateQuotaRequest {
        val builder = commonSettings.baseQuotaRequest(providerName())
        val orders = commonSettings.orders(campaignType, OrdersSubset.FULL)
        var currentValue = 1L
        val providersList = listOf(Pair(s3ProviderKey, s3Resources), Pair(nirvanaProviderKey, nirvanaResources),
            Pair(sandboxProviderKey, sandboxResources), Pair(mdbProviderKey, mdbResources))
        for ((providerKey, resources) in providersList) {
            for (resource in resources) {
                for (orderId in orders) {
                    builder
                        .addChange(CreateChange.builder()
                            .serviceKey(providerKey)
                            .orderId(orderId)
                            .resourceKey(resource.key)
                            .amount(Amount.builder()
                                .value(currentValue++)
                                .unit(resource.unit)
                                .build())
                            .build())
                }
            }
        }
        return builder.build()
    }

    override fun validateTestRequest(request: QuotaRequest,
                                     environment: Environment,
                                     campaignType: CampaignType): Result<String> {
        val requestedResources = mutableMapOf<Long, MutableMap<SegmentedResourceKey, RequestedAmount>>()
        request.changes.forEach{ change ->
            val key = SegmentedResourceKey(change.service.key, change.resource.key, change.segmentKeys)
            val amount = RequestedAmount(change.amount.value, Units.UNITS[change.amount.unit])
            requestedResources.computeIfAbsent(change.order.id) { mutableMapOf() }[key] = amount
        }
        val baseResources = mutableMapOf<Long, MutableMap<SegmentedResourceKey, RequestedAmount>>()
        request.baseResourceChanges.forEach { change ->
            val key = SegmentedResourceKey(change.baseResource.type.provider.key, change.baseResource.type.key,
                change.baseResource.segmentKeys)
            val amount = RequestedAmount(change.totalAmount.value, Units.UNITS[change.totalAmount.unit])
            baseResources.computeIfAbsent(change.bigOrder.id) { mutableMapOf() }[key] = amount
        }
        val expectedBaseResources = mutableMapOf<Long, MutableMap<SegmentedResourceKey, RequestedAmount>>()
        requestedResources.forEach { (orderId, resources) ->
            val requestedS3HddX2 = resources[SegmentedResourceKey(s3ProviderKey,
                "x2_hdd", emptySet())]!!
            if (requestedS3HddX2.unit != Units.Bytes.BYTES) {
                logger.error("Unexpected S3 HDD x2 unit: {}", requestedS3HddX2)
                return Result.failure(ErrorCollection.builder().addError("Unexpected S3 HDD x2 unit").build())
            }
            val requestedS3HddX3 = resources[SegmentedResourceKey(s3ProviderKey,
                "x3_hdd", emptySet())]!!
            if (requestedS3HddX3.unit != Units.Bytes.BYTES) {
                logger.error("Unexpected S3 HDD x3 unit: {}", requestedS3HddX3)
                return Result.failure(ErrorCollection.builder().addError("Unexpected S3 HDD x3 unit").build())
            }
            val requestedNirvanaHddX2 = resources[SegmentedResourceKey(nirvanaProviderKey,
                "s3_hdd_x2", emptySet())]!!
            if (requestedNirvanaHddX2.unit != Units.Bytes.BYTES) {
                logger.error("Unexpected Nirvana HDD x2 unit: {}", requestedNirvanaHddX2)
                return Result.failure(ErrorCollection.builder().addError("Unexpected Nirvana HDD x2 unit").build())
            }
            val requestedSandboxHddX2 = resources[SegmentedResourceKey(sandboxProviderKey,
                "s3_hdd_x2", emptySet())]!!
            if (requestedSandboxHddX2.unit != Units.Bytes.BYTES) {
                logger.error("Unexpected Sandbox HDD x2 unit: {}", requestedSandboxHddX2)
                return Result.failure(ErrorCollection.builder().addError("Unexpected Sandbox HDD x2 unit").build())
            }
            val requestedMdbHddX3 = resources[SegmentedResourceKey(mdbProviderKey,
                "s3_hdd_x3", emptySet())]!!
            if (requestedMdbHddX3.unit != Units.Bytes.BYTES) {
                logger.error("Unexpected MDB HDD x3 unit: {}", requestedMdbHddX3)
                return Result.failure(ErrorCollection.builder().addError("Unexpected MDB HDD x3 unit").build())
            }
            expectedBaseResources.computeIfAbsent(orderId) { mutableMapOf() }[SegmentedResourceKey(s3ProviderKey,
                "s3:x2_hdd", emptySet())] = RequestedAmount(toGibibytes(requestedS3HddX2.amount) +
                toGibibytes(requestedNirvanaHddX2.amount) + toGibibytes(requestedSandboxHddX2.amount),
                Units.Gibibytes.GIBIBYTES)
            expectedBaseResources.computeIfAbsent(orderId) { mutableMapOf() }[SegmentedResourceKey(s3ProviderKey,
                "s3:x3_hdd", emptySet())] = RequestedAmount(toGibibytes(requestedS3HddX3.amount) +
                toGibibytes(requestedMdbHddX3.amount), Units.Gibibytes.GIBIBYTES)
        }
        logger.info { "Expected ${providerName()} base resources: $expectedBaseResources" }
        logger.info { "Actual ${providerName()} base resources: $baseResources" }
        val match = baseResources == expectedBaseResources
        return if (match) {
            Result.success(providerName())
        } else {
            Result.failure(ErrorCollection.builder().addError("Base resource mismatch for ${providerName()}").build())
        }
    }

    override fun providerName() = "S3"

    private fun toGibibytes(bytes: Long) = bytes / (1024L * 1024L * 1024L)

}
