package ru.yandex.dispenser.validation.providers.sandbox;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.dispenser.validation.client.CampaignType;
import ru.yandex.dispenser.validation.client.Environment;
import ru.yandex.dispenser.validation.client.ErrorCollection;
import ru.yandex.dispenser.validation.client.OrdersSubset;
import ru.yandex.dispenser.validation.client.Result;
import ru.yandex.dispenser.validation.client.model.Amount;
import ru.yandex.dispenser.validation.client.model.CreateChange;
import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest;
import ru.yandex.dispenser.validation.client.model.QuotaRequest;
import ru.yandex.dispenser.validation.client.model.Units;
import ru.yandex.dispenser.validation.providers.CommonSettings;
import ru.yandex.dispenser.validation.providers.ProviderValidator;
import ru.yandex.dispenser.validation.providers.RequestedAmount;
import ru.yandex.dispenser.validation.providers.RequestedResource;
import ru.yandex.dispenser.validation.providers.SegmentedResourceKey;

/**
 * Sandbox validator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class SandboxValidator implements ProviderValidator {

    private static final Logger LOG = LoggerFactory.getLogger(SandboxValidator.class);

    private static final String PROVIDER_KEY = "sandbox";

    private final CommonSettings commonSettings;
    private final Map<String, List<RequestedResource>> resourcesBySegment;

    public SandboxValidator(CommonSettings commonSettings) {
        this.commonSettings = commonSettings;
        this.resourcesBySegment = Map.of(
                "sandbox_generic", List.of(
                        new RequestedResource("cpu_linux_hdd", Units.Cores.CORES),
                        new RequestedResource("cpu_linux_ssd", Units.Cores.CORES),
                        new RequestedResource("cpu_macos", Units.Cores.CORES),
                        new RequestedResource("cpu_windows", Units.Cores.CORES)
                ),
                "sandbox_browser", List.of(
                        new RequestedResource("cpu_linux_hdd", Units.Cores.CORES),
                        new RequestedResource("cpu_linux_ssd", Units.Cores.CORES),
                        new RequestedResource("cpu_macos", Units.Cores.CORES),
                        new RequestedResource("cpu_windows", Units.Cores.CORES)
                ),
                "sandbox_yabs", List.of(
                        new RequestedResource("cpu_linux_hdd", Units.Cores.CORES),
                        new RequestedResource("cpu_linux_ssd", Units.Cores.CORES)
                ),
                "sandbox_mobile_monorepo", List.of(
                        new RequestedResource("cpu_macos", Units.Cores.CORES)
                )
        );
    }

    @Override
    public CreateQuotaRequest prepareTestRequest(Environment environment, CampaignType campaignType) {
        CreateQuotaRequest.Builder builder = commonSettings.baseQuotaRequest(providerName());
        List<Long> orders = commonSettings.orders(campaignType, OrdersSubset.FULL);
        long currentValue = 1L;
        for (String segmentKey : resourcesBySegment.keySet()) {
            for (RequestedResource resource : resourcesBySegment.get(segmentKey)) {
                for (Long orderId : orders) {
                    long value = currentValue++;
                    builder.addChange(CreateChange.builder()
                            .serviceKey(PROVIDER_KEY)
                            .orderId(orderId)
                            .resourceKey(resource.getKey())
                            .addSegmentKey(segmentKey)
                            .amount(Amount.builder()
                                    .value(value)
                                    .unit(resource.getUnit())
                                    .build())
                            .build());
                }
            }
        }
        return builder.build();
    }

    @Override
    public Result<String> validateTestRequest(QuotaRequest request, Environment environment,
                                              CampaignType campaignType) {
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> requestedResources = new HashMap<>();
        request.getChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getService().getKey(),
                    change.getResource().getKey(), change.getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getAmount().getValue(),
                    Units.UNITS.get(change.getAmount().getUnit()));
            requestedResources.computeIfAbsent(change.getOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> baseResources = new HashMap<>();
        request.getBaseResourceChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getBaseResource().getType()
                    .getProvider().getKey(), change.getBaseResource().getType().getKey(),
                    change.getBaseResource().getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getTotalAmount().getValue(),
                    Units.UNITS.get(change.getTotalAmount().getUnit()));
            baseResources.computeIfAbsent(change.getBigOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> expectedBaseResources = new HashMap<>();
        for (Map.Entry<Long, Map<SegmentedResourceKey, RequestedAmount>> entry : requestedResources.entrySet()) {
            Long orderId = entry.getKey();
            Map<SegmentedResourceKey, RequestedAmount> resourcesMap = entry.getValue();
            Long baseCpuLinuxHdd = null;
            Long baseCpuLinuxSsd = null;
            Long baseCpuMacos = null;
            Long baseCpuWindows = null;
            for (String segmentKey : resourcesBySegment.keySet()) {
                RequestedAmount requestedCpuLinuxHdd = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                        "cpu_linux_hdd", Set.of(segmentKey)));
                RequestedAmount requestedCpuLinuxSsd = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                        "cpu_linux_ssd", Set.of(segmentKey)));
                RequestedAmount requestedCpuMacos = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                        "cpu_macos", Set.of(segmentKey)));
                RequestedAmount requestedCpuWindows = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                        "cpu_windows", Set.of(segmentKey)));
                if (requestedCpuLinuxHdd != null) {
                    if (!requestedCpuLinuxHdd.getUnit().equals(Units.Cores.PERMILLE_CORES)) {
                        LOG.error("Unexpected CPU unit: {}", requestedCpuLinuxHdd);
                        return Result.failure(ErrorCollection.builder().addError("Unexpected CPU unit").build());
                    }
                    baseCpuLinuxHdd = baseCpuLinuxHdd != null
                            ? baseCpuLinuxHdd + requestedCpuLinuxHdd.getAmount() : requestedCpuLinuxHdd.getAmount();
                }
                if (requestedCpuLinuxSsd != null) {
                    if (!requestedCpuLinuxSsd.getUnit().equals(Units.Cores.PERMILLE_CORES)) {
                        LOG.error("Unexpected CPU unit: {}", requestedCpuLinuxSsd);
                        return Result.failure(ErrorCollection.builder().addError("Unexpected CPU unit").build());
                    }
                    baseCpuLinuxSsd = baseCpuLinuxSsd != null
                            ? baseCpuLinuxSsd + requestedCpuLinuxSsd.getAmount() : requestedCpuLinuxSsd.getAmount();
                }
                if (requestedCpuMacos != null) {
                    if (!requestedCpuMacos.getUnit().equals(Units.Cores.PERMILLE_CORES)) {
                        LOG.error("Unexpected CPU unit: {}", requestedCpuMacos);
                        return Result.failure(ErrorCollection.builder().addError("Unexpected CPU unit").build());
                    }
                    baseCpuMacos = baseCpuMacos != null
                            ? baseCpuMacos + requestedCpuMacos.getAmount() : requestedCpuMacos.getAmount();
                }
                if (requestedCpuWindows != null) {
                    if (!requestedCpuWindows.getUnit().equals(Units.Cores.PERMILLE_CORES)) {
                        LOG.error("Unexpected CPU unit: {}", requestedCpuWindows);
                        return Result.failure(ErrorCollection.builder().addError("Unexpected CPU unit").build());
                    }
                    baseCpuWindows = baseCpuWindows != null
                            ? baseCpuWindows + requestedCpuWindows.getAmount() : requestedCpuWindows.getAmount();
                }
            }
            if (baseCpuLinuxHdd != null) {
                expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                        .put(new SegmentedResourceKey(PROVIDER_KEY, "sandbox:cpu_linux_hdd",
                                Set.of()), new RequestedAmount(baseCpuLinuxHdd, Units.Cores.PERMILLE_CORES));
            }
            if (baseCpuLinuxSsd != null) {
                expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                        .put(new SegmentedResourceKey(PROVIDER_KEY, "sandbox:cpu_linux_ssd",
                                Set.of()), new RequestedAmount(baseCpuLinuxSsd, Units.Cores.PERMILLE_CORES));
            }
            if (baseCpuMacos != null) {
                expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                        .put(new SegmentedResourceKey(PROVIDER_KEY, "sandbox:cpu_macos",
                                Set.of()), new RequestedAmount(baseCpuMacos, Units.Cores.PERMILLE_CORES));
            }
            if (baseCpuWindows != null) {
                expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                        .put(new SegmentedResourceKey(PROVIDER_KEY, "sandbox:cpu_windows",
                                Set.of()), new RequestedAmount(baseCpuWindows, Units.Cores.PERMILLE_CORES));
            }
        }
        LOG.info("Expected {} base resources: {}", providerName(), expectedBaseResources);
        LOG.info("Actual {} base resources: {}", providerName(), baseResources);
        boolean match = baseResources.equals(expectedBaseResources);
        if (match) {
            return Result.success(providerName());
        } else {
            return Result.failure(ErrorCollection.builder().addError("Base resource mismatch for " + providerName())
                    .build());
        }
    }

    @Override
    public String providerName() {
        return "Sandbox";
    }

}
