package ru.yandex.dispenser.validation.providers.solomon;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.dispenser.validation.client.CampaignType;
import ru.yandex.dispenser.validation.client.Environment;
import ru.yandex.dispenser.validation.client.ErrorCollection;
import ru.yandex.dispenser.validation.client.OrdersSubset;
import ru.yandex.dispenser.validation.client.Result;
import ru.yandex.dispenser.validation.client.model.Amount;
import ru.yandex.dispenser.validation.client.model.BaseResourceChange;
import ru.yandex.dispenser.validation.client.model.CreateChange;
import ru.yandex.dispenser.validation.client.model.CreateQuotaRequest;
import ru.yandex.dispenser.validation.client.model.QuotaRequest;
import ru.yandex.dispenser.validation.client.model.Units;
import ru.yandex.dispenser.validation.providers.CommonSettings;
import ru.yandex.dispenser.validation.providers.ProviderValidator;
import ru.yandex.dispenser.validation.providers.RequestedAmount;
import ru.yandex.dispenser.validation.providers.RequestedResource;
import ru.yandex.dispenser.validation.providers.SegmentedResourceKey;

/**
 * Solomon validator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class SolomonValidator implements ProviderValidator {

    private static final Logger LOG = LoggerFactory.getLogger(SolomonValidator.class);

    private static final String PROVIDER_KEY = "solomon";

    private final CommonSettings commonSettings;
    private final RequestedResource storedMetrics;
    private final RequestedResource writeFlow;
    private final RequestedResource alertsCount;
    private final Set<String> segmentKeys;
    private final Set<String> reducedOrderSubsetSegmentKeys;

    public SolomonValidator(CommonSettings commonSettings) {
        this.commonSettings = commonSettings;
        this.storedMetrics = new RequestedResource("metrics_stored_count", Units.Count.COUNT);
        this.writeFlow = new RequestedResource("metrics_write_flow_segmented", Units.Count.COUNT);
        this.alertsCount = new RequestedResource("alerts_count", Units.Count.COUNT);
        this.segmentKeys = Set.of("KLG", "SAS", "VLA", "IVA", "MYT");
        this.reducedOrderSubsetSegmentKeys = Set.of("KLG");
    }

    @Override
    public CreateQuotaRequest prepareTestRequest(Environment environment, CampaignType campaignType) {
        CreateQuotaRequest.Builder builder = commonSettings.baseQuotaRequest(providerName());
        List<Long> orders = commonSettings.orders(campaignType, OrdersSubset.FULL);
        List<Long> ordersReducedSubset = commonSettings.orders(campaignType, OrdersSubset.REDUCED);
        long currentValue = 1L;
        for (Long orderId : orders) {
            long value = currentValue++;
            builder.addChange(CreateChange.builder()
                    .serviceKey(PROVIDER_KEY)
                    .orderId(orderId)
                    .resourceKey(storedMetrics.getKey())
                    .amount(Amount.builder()
                            .value(value)
                            .unit(storedMetrics.getUnit())
                            .build())
                    .build());
        }
        for (Long orderId : orders) {
            long value = currentValue++;
            builder.addChange(CreateChange.builder()
                    .serviceKey(PROVIDER_KEY)
                    .orderId(orderId)
                    .resourceKey(alertsCount.getKey())
                    .amount(Amount.builder()
                            .value(value)
                            .unit(alertsCount.getUnit())
                            .build())
                    .build());
        }
        for (String segmentKey : segmentKeys) {
            List<Long> selectedOrders = reducedOrderSubsetSegmentKeys.contains(segmentKey)
                    ? ordersReducedSubset : orders;
            for (Long orderId : selectedOrders) {
                long value = currentValue++;
                builder.addChange(CreateChange.builder()
                        .serviceKey(PROVIDER_KEY)
                        .orderId(orderId)
                        .resourceKey(writeFlow.getKey())
                        .addSegmentKey(segmentKey)
                        .amount(Amount.builder()
                                .value(value)
                                .unit(writeFlow.getUnit())
                                .build())
                        .build());
            }
        }
        return builder.build();
    }

    @Override
    public Result<String> validateTestRequest(QuotaRequest request, Environment environment,
                                              CampaignType campaignType) {
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> requestedResources = new HashMap<>();
        request.getChanges().forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getService().getKey(),
                    change.getResource().getKey(), change.getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getAmount().getValue(),
                    Units.UNITS.get(change.getAmount().getUnit()));
            requestedResources.computeIfAbsent(change.getOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> baseResources = new HashMap<>();
        Objects.requireNonNullElse(request.getBaseResourceChanges(), Set.<BaseResourceChange>of()).forEach(change -> {
            SegmentedResourceKey key = new SegmentedResourceKey(change.getBaseResource().getType()
                    .getProvider().getKey(), change.getBaseResource().getType().getKey(),
                    change.getBaseResource().getSegmentKeys());
            RequestedAmount amount = new RequestedAmount(change.getTotalAmount().getValue(),
                    Units.UNITS.get(change.getTotalAmount().getUnit()));
            baseResources.computeIfAbsent(change.getBigOrder().getId(), k -> new HashMap<>()).put(key, amount);
        });
        Map<Long, Map<SegmentedResourceKey, RequestedAmount>> expectedBaseResources = new HashMap<>();
        for (Map.Entry<Long, Map<SegmentedResourceKey, RequestedAmount>> entry : requestedResources.entrySet()) {
            Long orderId = entry.getKey();
            Map<SegmentedResourceKey, RequestedAmount> resourcesMap = entry.getValue();
            for (String segmentKey : segmentKeys) {
                RequestedAmount requestedWriteFlow = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                        "metrics_write_flow_segmented", Set.of(segmentKey)));
                if (requestedWriteFlow != null) {
                    expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                            .put(new SegmentedResourceKey(PROVIDER_KEY, "solomon:metrics_write_flow_segmented",
                                    Set.of(segmentKey)), requestedWriteFlow);
                }
            }
            RequestedAmount requestedStoredCount = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                    "metrics_stored_count", Set.of()));
            expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                    .put(new SegmentedResourceKey(PROVIDER_KEY, "solomon:metrics_stored_count",
                            Set.of()), requestedStoredCount);
            RequestedAmount requestedAlertsCount = resourcesMap.get(new SegmentedResourceKey(PROVIDER_KEY,
                    "alerts_count", Set.of()));
            expectedBaseResources.computeIfAbsent(orderId, k -> new HashMap<>())
                    .put(new SegmentedResourceKey(PROVIDER_KEY, "solomon:alerts_count",
                            Set.of()), requestedAlertsCount);
        }
        LOG.info("Expected {} base resources: {}", providerName(), expectedBaseResources);
        LOG.info("Actual {} base resources: {}", providerName(), baseResources);
        boolean match = baseResources.equals(expectedBaseResources);
        if (match) {
            return Result.success(providerName());
        } else {
            return Result.failure(ErrorCollection.builder().addError("Base resource mismatch for " + providerName())
                    .build());
        }
    }

    @Override
    public String providerName() {
        return "Solomon";
    }
}
