import psycopg2
import requests
import logging
import json
from library.python.vault_client.instances import Production as VaultClient
from psycopg2.extras import DictCursor

TOKENS = {
    'robot-dispenser': {'ver': 'ver-01dj5e0vzsw63m6mv15etzmv13', 'key': 'oauth_token'},
    'robot-tracker': {'ver': 'ver-01d54mks4epb6phqh03tkjnk3d', 'key': 'tracker_oauth'},
    'bot-prod': {'ver': 'ver-01dhtvq5cgg0qe5bjncds6f158', 'key': 'oauth_token'},
    'testing_db': {'ver': 'ver-01emr5b2rw8prhkz994j46jxmz', 'key': 'password'},
    'prod_db': {'ver': 'ver-01emr5cj2vkznrnzmz0g7ncnqv', 'key': 'password'},
    'robot-dispenser-d-prod': {'ver': 'ver-01fjbszn80bvg5jd765r9ts9qm', 'key': 'oauth_token_prod'},
    'robot-dispenser-d-test': {'ver': 'ver-01fjbszn80bvg5jd765r9ts9qm', 'key': 'oauth_token_test'},
}


def get_secret_value(ver, key):
    vault_client = VaultClient()
    return vault_client.get_version(ver)['value'][key]


def get_token(key):
    params = TOKENS[key]
    return get_secret_value(params['ver'], params['key'])


def prod_password():
    return get_token("prod_db")


def testing_password():
    return get_token("testing_db")


def robot_dispenser_token():
    return get_token("robot-dispenser")


def robot_dispenser_d_token(env):
    return get_token("robot-dispenser-d-" + env)


def common_testing_connection():
    return psycopg2.connect(f"""
        host=man-65p53fqgfjg9vz9o.db.yandex.net,sas-sryxv84a8ba7m1dk.db.yandex.net,vla-998kbuokjq8nzbxt.db.yandex.net
        port=6432
        dbname=dispenser_common_testing
        user=dispenser
        password={testing_password()}
        target_session_attrs=read-write
        sslmode=verify-full
    """)


def common_prod_connection():
    return psycopg2.connect(f"""
        host=man-f10sh5xrioku6izp.db.yandex.net,sas-z3btrew8mc1moza1.db.yandex.net,vla-92csqm2om2u3rww3.db.yandex.net
        port=6432
        dbname=dispenser_common_production
        user=dispenser
        password={prod_password()}
        target_session_attrs=read-write
        sslmode=verify-full
    """)


def dict_cursor(connection):
    return connection.cursor(cursor_factory=DictCursor)


def configure_logging(filename):
    logging.basicConfig(filename=filename, filemode='w', level=logging.DEBUG,
                        format='%(asctime)s - [%(levelname)s] - %(message)s')


class Response:
    def __init__(self, body, status):
        self.body = body
        self.status = status


class Dispenser:

    def __init__(self, prod, cluster=''):
        self.token = robot_dispenser_token()
        if prod:
            self.host = 'https://dispenser.yandex-team.ru/'
        else:
            self.host = 'https://dispenser.test.yandex-team.ru/'

        if cluster != '':
            self.host += cluster + '/'

    def prepare_headers(self, additional=None):
        if additional is None:
            additional = {}
        return {
            "Authorization": f"OAuth {self.token}",
            **additional
        }

    def get_requests(self, **params):
        headers = self.prepare_headers()
        resp = requests.get(self.host + 'api/v1/quota-requests', headers=headers, params=params)
        res = resp.json()
        if resp.status_code != 200:
            return Response(res, resp.status_code)
        return Response(res['result'], resp.status_code)

    def set_quota_state(self, body):
        headers = self.prepare_headers({
            "Content-Type": "application/json"
        })
        resp = requests.patch(self.host + 'api/v1/resource-preorder/quotaState', data=json.dumps(body), headers=headers)
        res = resp.json()
        return Response(res, resp.status_code)

    def set_quota_state_optional(self, body):
        headers = self.prepare_headers({
            "Content-Type": "application/json"
        })
        resp = requests.patch(self.host + 'api/v1/resource-preorder/quotaStateOptional', data=json.dumps(body),
                              headers=headers)
        res = resp.json()
        return Response(res, resp.status_code)

    def allocate_provider(self, request_id, provider):
        headers = self.prepare_headers()
        resp = requests.post(self.host + f'api/v1/resource-preorder/{request_id}/provider-allocation/{provider}',
                             headers=headers)
        res = resp.json()
        return Response(res, resp.status_code)


class D:

    def __init__(self, prod):
        self.token = robot_dispenser_d_token('prod' if prod else 'test')
        if prod:
            self.host = 'https://d-api.yandex-team.ru/'
        else:
            self.host = 'https://d.test.yandex-team.ru/'

    def _prepare_headers(self, additional=None):
        if additional is None:
            additional = {}
        return {
            "Authorization": f"OAuth {self.token}",
            **additional
        }

    def _log_on_error(self, resp):
        if resp.status_code > 299:
            logging.error(f"Got {resp.status_code}: {resp.text}")

    def get_all_providers(self):
        headers = self._prepare_headers()
        resp = requests.get(self.host + "api/v1/providers", headers=headers, params={'limit': 100})
        self._log_on_error(resp)
        res = resp.json()
        return Response(res, resp.status_code)

    def get_provider_resources(self, provider_id):
        headers = self._prepare_headers()
        resp = requests.get(self.host + f"api/v1/providers/{provider_id}/resources", headers=headers,
                            params={"limit": 100})
        self._log_on_error(resp)
        res = resp.json()
        return Response(res, resp.status_code)
