# coding: utf-8
from rest_framework.exceptions import APIException
from rest_framework import status


class RestApiError(APIException):
    """
    Базовая ошибка. Не должна выбрасываться сама по себе, вместо этого нужно использовать наследников.
    У наследников должен быть определен status_code.
    """
    detail = None

    def __init__(self, detail=None, errors=None):
        """
        @param errors: словарь с информацией о деталях ошибки: ключ – строка, индентификатор детали,
                       значение - строка, пояснение, в чём проблема с этой деталью.
        @param detail: строка, сообщение об ошибке

        Когда нужно вернуть текстовое сообщение для ожидаемых ошибок, следует стараться помещать его в errors.
        detail - для unexpected ошибок.
        """
        self.detail = detail or self.detail
        self.errors = errors or []
        assert hasattr(self, 'status_code'), 'Class %s must have "status_code" attribute' % type(self)

    def __repr__(self):
        return '%s(detail=%s, errors=%s)' % (
            self.__class__.__name__,
            repr(self.detail),
            repr(self.errors),
        )


class UnhandledException(RestApiError):
    status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
    detail = 'Unhandled exception'


class BadRequestError(RestApiError):
    status_code = status.HTTP_400_BAD_REQUEST
    detail = 'Bad request'


class UnprocessableEntityError(RestApiError):
    status_code = 422  # Unprocessable entity
    detail = 'Validation Failed'


class ResourceIsMissing(RestApiError):
    """
    Отсутствует ресурс или его необходимый компонент.
    """
    status_code = status.HTTP_404_NOT_FOUND
    detail = 'Resource or its component is missing'
