# coding: utf-8
import logging
import traceback

from django.http import Http404
from django.conf import settings
from django.shortcuts import redirect
from django.utils.functional import cached_property
from django_tools_log_context import request_context, request_profiler
from rest_framework.generics import GenericAPIView
from rest_framework.response import Response
from rest_framework.exceptions import APIException, AuthenticationFailed
from rest_framework.permissions import BasePermission
from rest_framework.authentication import BaseAuthentication
from dir_data_sync.org_ctx import org_ctx

from intranet.dogma.dogma.core.models import Repo
from intranet.dogma.dogma.api.errors import UnhandledException
from intranet.dogma.dogma.api.utils import (to_positive_int, make_absolute_url, RepoHolder,
                             RepositoryError, Http302)

logger = logging.getLogger(__file__)


class HiddenSourcePermission(BasePermission):
    def has_object_permission(self, request, view, obj):
        if isinstance(obj, Repo):
            return not obj.source.hidden

        return True


class BlackboxAuthentication(BaseAuthentication):
    def authenticate(self, request):
        if request.yauser.is_authenticated():
            return (request.yauser, None)
        else:
            raise AuthenticationFailed('Invalid token/ticket')


class APIView(GenericAPIView):
    """
    Базовый класс для всех API View
    """
    authentication_classes = (BlackboxAuthentication,)
    permission_classes = (HiddenSourcePermission,)

    def get_paginate_by(self):
        paginate_by = self.request.GET.get('per_page', settings.DOGMA_API_DEFAULT_PAGINATE_BY)
        paginate_by = to_positive_int(paginate_by, 'per_page', strict_positive=True)
        return min(paginate_by, settings.DOGMA_API_MAX_PAGINATE_BY)

    def handle_exception(self, exc):
        """
        Обработать исключения из хэндлера запроса
        @param exc: объект исключения
        """
        logger.info('API error occured: %s', repr(exc))

        if isinstance(exc, Http404):
            return Response(
                {
                    'message': repr(exc)
                },
                status=404,
                exception=True,
            )

        if isinstance(exc, APIException):
            # Штатная ошибка или ошибка rest_framework
            data = {
                'message': exc.detail,
            }
            if getattr(exc, 'errors', None):
                data['errors'] = exc.errors

            logger.debug('Responded with "%s"', data)
            return Response(data, status=exc.status_code)

        elif isinstance(exc, Http302):
            resp = redirect(exc.location)
            if exc.internal:
                resp['X-Accel-Redirect'] = exc.location

            return resp

        else:
            # Ошибка 500
            logger.exception('Error happened while handling request')
            data = {
                'message': UnhandledException.detail,
            }
            if settings.DEBUG:
                data['errors'] = {
                    'traceback': traceback.format_exc(),
                }

            logger.debug('Responded with "%s"', repr(data))
            return Response(data, status=UnhandledException.status_code, exception=True)

    def successful_response(self):
        return Response(status=200)

    def dispatch(self, request, *args, **kwargs):
        with org_ctx(self.request.org):
            with request_context(request, endpoint=self), request_profiler(request, threshold=5):
                return super(APIView, self).dispatch(request, *args, **kwargs)


class RepoAPIView(APIView):
    @cached_property
    def repo(self):
        """
        Возвращает объект RepoHolder, если во вьюху переданы source, owner и name
        """

        id_ = RepoHolder(self.kwargs['source'], self.kwargs['owner'], self.kwargs['name'])

        try:
            self.check_object_permissions(self.request, id_.model)
        except Repo.DoesNotExist:
            raise Http404('Unknown repo: %s, %s, %s' % (self.kwargs['source'],
                                                        self.kwargs['owner'],
                                                        self.kwargs['name']))

        return id_

    @cached_property
    def raw(self):
        try:
            return self.repo.get_raw(self.request)
        except RepositoryError as exception:
            raise Http404(str(exception))

    @cached_property
    def model(self):
        try:
            return self.repo.model
        except Repo.DoesNotExist:
            raise Http404('Unknown repo')


class ListModelMixin(object):
    """
    Кастомный ListModelMixin, отличается тем, что паджинация выполнена в стиле Github.

    paginate_queryset должен возвращать не объект django.core.paginator.Page,
    а Queryset вместе с параметрами для добавления ссылок в хедеры.
    Параметры должны быть в списке вида:
    [
        ('next', [('page', 3)]),
        ('last', [('page', 5)]),
        ('first', []),
    ]
    """
    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        queryset, pagination_data = self.paginate_queryset(queryset)

        serializer = self.get_serializer(queryset, many=True)
        response = Response(serializer.data)
        links = ['<%s>; rel="%s"' % (
            make_absolute_url(self.request, parameters=params),
            key
        ) for (key, params) in pagination_data]
        response['Link'] = ', '.join(links)
        return response
