# coding: utf-8
import shlex
from dateutil import parser


def get_parsed_query(q):
    """
    Получаем из строки запроса словарь
    вида {параметр: условие}, условие
    всегда возвращаем в виде списка

    Примеры:
    >>> q = 'author:smosker+queues:AUDTOOL+tickets:WIKI-1273,WIKI-3456'
    >>> get_parsed_query(q)
    {'tickets': ['WIKI-1273', 'WIKI-3456'], 'queues': ['AUDTOOL'], 'author': ['smosker']}

    :param q: basestring
    :rtype: dict(basestring: list)
    """
    parsed_query = {}

    request_args = shlex.split(q)
    for query in request_args:
        param, term = query.split(':', 1)
        result_term = term.split(',')
        parsed_query[param] = result_term

    return parsed_query


def get_response_params(parsed_query, available_queries, additional_parameters):
    """
    Формируем запрос для базы, запрос формируется только по параметрам,
    которые являются ключами в available_queries
    Так же возвращает дополнительные параметры, если они были
    переданы в запросе
    """
    include_query = {}
    exclude_query = {}
    parameters = {}

    for param, term in parsed_query.items():
        if param in additional_parameters:
            parameters[param] = term
            continue

        known_query = available_queries.get(param)
        if not known_query:
            continue

        query = known_query.get('field', param)

        query_suffix = known_query.get('query_suffix', 'in')
        values_map = known_query.get('values_map')
        list_values = known_query.get('list')
        apply_on_each = known_query.get('apply_on_each')
        apply_bulk = known_query.get('apply_bulk')

        if values_map:
            term = [values_map.get(elem, elem) for elem in term]

        if query_suffix:
            query = '{}__{}'.format(query, query_suffix)

        if apply_on_each:
            term = [apply_on_each(term) for term in term]

        if apply_bulk:
            term = apply_bulk(term)

        if list_values is False:
            term = ' '.join(term)

        exclude = known_query.get('exclude')
        if exclude:
            exclude_query[query] = term
        else:
            include_query[query] = term

    return include_query, exclude_query, parameters


def prepare_date(term):
    """
    Преобразуем дату в виде
    строки в datetime объект

    :param term: basestring
    :rtype: datetime
    """
    return parser.parse(term)
