# coding: utf-8
import re
import waffle

from rest_framework import serializers
from django.conf import settings

from intranet.dogma.dogma.waffle_switches import REWRITE_GITHUB_LINKS

github_regex = re.compile(r'^https?:\/\/github\.yandex-team\.ru')


class GitNodeSerializer(serializers.Serializer):
    url = serializers.URLField()
    sha = serializers.CharField()


class RepoSerializer(serializers.Serializer):
    name = serializers.CharField()
    url = serializers.SerializerMethodField()

    def get_url(self, data):
        url = data.get('url', '')
        if waffle.switch_is_active(REWRITE_GITHUB_LINKS):
            url = github_regex.sub('https://gitlab.yandex-team.ru', url)
        return url


class SignatureSerializer(serializers.Serializer):
    """
    Пользователь гита.
    """
    login = serializers.CharField()
    name = serializers.CharField()
    email = serializers.CharField()
    date = serializers.DateTimeField(format='%Y-%m-%dT%H:%M:%S')
    if not settings.IS_BUSINESS:
        uid = serializers.CharField()
    is_trusted_login = serializers.BooleanField()


class CommitMetaSerializer(serializers.Serializer):
    url = serializers.URLField()
    author = SignatureSerializer()
    committer = SignatureSerializer()
    message = serializers.CharField()
    tree = GitNodeSerializer()
    tickets = serializers.ListField(child=serializers.CharField(), required=False)
    queues = serializers.ListField(child=serializers.CharField(), required=False)


class StatsSerializer(serializers.Serializer):
    additions = serializers.IntegerField()
    deletions = serializers.IntegerField()
    total = serializers.IntegerField()


class FileSerializer(serializers.Serializer):
    sha = serializers.CharField()
    filename = serializers.CharField()
    status = serializers.CharField()
    additions = serializers.IntegerField()
    deletions = serializers.IntegerField()
    changes = serializers.IntegerField()
    raw_url = serializers.URLField()
    blob_url = serializers.URLField()
    contents_url = serializers.URLField()
    patch = serializers.CharField(required=False)


class CommitDogmaSerializer(serializers.Serializer):
    id = serializers.CharField()
    html_url = serializers.SerializerMethodField()

    def get_html_url(self, data):
        html_url = data.get('html_url', '')
        if waffle.switch_is_active(REWRITE_GITHUB_LINKS):
            html_url = github_regex.sub('https://gitlab.yandex-team.ru', html_url)
        return html_url


class CommitSerializer(serializers.Serializer):
    """
    Информация о коммите для списка коммитов
    https://developer.github.com/v3/repos/commits/
    """
    url = serializers.URLField()
    sha = serializers.CharField()
    commit = CommitMetaSerializer()
    parents = GitNodeSerializer(many=True)
    _dogma = CommitDogmaSerializer()


class MediumDetailedCommitSerializer(CommitSerializer):
    """
    Информация
    """
    stats = StatsSerializer(required=False)


class DetailedCommitSerializer(MediumDetailedCommitSerializer):
    """
    Информация о коммите для получения одного
    https://developer.github.com/v3/repos/commits/
    """
    stats = StatsSerializer(required=False)
    files = FileSerializer(many=True, required=False)
    branch_name = serializers.ListSerializer(child=serializers.CharField())
    repo = RepoSerializer()
