# coding: utf-8

from rest_framework import serializers
from rest_framework.reverse import reverse

from intranet.dogma.dogma.core.models import Repo


class OwnerSerializer(serializers.Serializer):
    """
    Сериализует строку логина в объект
    {
        'login': логин,
        'repos_url': адрес всех репозиториев в API,
    }
    """

    def get_attribute(self, repository):
        return {
            'login': '%s:%s' % (repository.source.code, repository.owner),
            'repos_url': reverse(
                'api:user_repositories',
                args=['users', repository.source.code, repository.owner],
                request=self.context['request'],
            ),
        }

    login = serializers.CharField()
    repos_url = serializers.URLField()


class RepositorySerializer(serializers.ModelSerializer):
    owner = OwnerSerializer()
    created_at = serializers.DateTimeField(source='created', format='%Y-%m-%dT%H:%M:%SZ')
    updated_at = serializers.DateTimeField(source='modified', format='%Y-%m-%dT%H:%M:%SZ')
    default_branch = serializers.CharField()
    is_public = serializers.BooleanField()

    class Meta:
        model = Repo
        fields = (
            'id', 'name', 'full_name', 'description',
            'created_at', 'updated_at', 'owner', 'default_branch',
            'is_public',
        )

    def to_representation(self, obj):
        defaults = super(RepositorySerializer, self).to_representation(obj)

        args = [obj.source.code, obj.owner, obj.name]
        geturl = lambda name: reverse(name, args=args, request=self.context['request'])

        defaults['url'] = geturl('api:repository')
        defaults['commits_url'] = geturl('api:commits')
        defaults['branches_url'] = geturl('api:branches')
        defaults['contributors_url'] = geturl('api:contributors')
        defaults['languages_url'] = geturl('api:languages')

        defaults['_dogma'] = {
            'source': obj.source.code,
            'html_url': obj.source.crawler.get_repo_url(obj),
        }

        return defaults
