# coding: utf-8
from hashlib import sha1
from urllib.parse import urlparse, urlunparse

from django.core.cache import cache
from django.utils.http import urlencode
from django.utils.functional import cached_property
from django.http import Http404

from intranet.dogma.dogma.api.errors import UnprocessableEntityError
from intranet.dogma.dogma.core.models import Repo, Clone
from intranet.dogma.dogma.core.utils import get_repository_model
from intranet.dogma.dogma.core.utils import get_current_node


def to_positive_int(number, param_name, strict_positive=False):
    """
    Делает int от числа и требует его положительность. В случае ошибки кидает стандартный эксепшн.
    """

    try:
        number = int(number)
    except (TypeError, ValueError):
        pass
    else:
        if number >= 0 and not (strict_positive and number == 0):
            return number

    raise UnprocessableEntityError(detail='Wrong page number', errors=[
        {'field': param_name, 'error': 'Invalid value: "%s"' % number}
    ])


def make_absolute_url(request, path=None, parameters=None):
    url = request.build_absolute_uri(path or request.path)

    if parameters:
        if isinstance(parameters, list):
            parameters = urlencode(parameters)
        url += '?' + parameters

    return url


class RepositoryError(Exception):
    pass


class Http302(Exception):
    def __init__(self, location, internal=False):
        self.location = location
        self.internal = internal


class RepoHolder(object):
    """
    ID репозиториев имеют вид {source}:{owner}/{name},
    Этот класс их сладывает в один объект для удобства
    """
    def __init__(self, source, owner, name):
        self.source = source
        self.owner = owner
        self.name = name

    @property
    def args(self):
        """
        Хелпер, чтобы не перечислять вручную.
        """
        return [self.source, self.owner, self.name]

    @cached_property
    def model(self):
        return Repo.objects.get(source__code=self.source, owner=self.owner, name=self.name)

    def get_latest_clone(self, clones):
        return max(clones, key=lambda x: x.modified)

    def get_raw(self, request):
        """
        Объект репозитория pygit или редирект
        """
        key = "clones_routs_%s_%s_%s" % (self.source, self.owner, self.name)
        route = cache.get(key, None)
        current_node = get_current_node()
        redir = lambda clone: Http302(self.get_redir_url(clone, request), internal=True)

        if route is None:
            try:
                clones = list(self.model.clones.select_related('node').all())
                if not clones:
                    raise Clone.DoesNotExist()

            except (Repo.DoesNotExist, Clone.DoesNotExist):
                    raise RepositoryError('Repository is not found')

            latest = self.get_latest_clone(clones).modified

            for clone in clones:
                # если пользователь пришел на машину с
                # свежим репозиторием, то редирект не нужен
                if clone.node == current_node and clone.modified == latest:
                    path = clone.path
                    break
            else:
                clone = self.choose_clone(clones, request)
                clone._state.db = None  # для репликейтеда
                cache.set(key, clone)

                raise redir(clone)
        else:
            if route.node != current_node:
                raise redir(route)

        if clone.status != Clone.STATUSES.active:
            # Это нештатная ситуация
            # Надо удалить такой клон вручную из базы.
            raise Http404('Clone is not active')

        return get_repository_model(clone)

    def get_redir_url(self, clone, request):
        url = request.build_absolute_uri()
        _, _, path, params, query, fragment = urlparse(url)
        return urlunparse(('', '',
                           '/internal_redirect/%s%s' % (clone.node.hostname, path),
                           params, query, fragment))

    def choose_clone(self, clones, request):
        last = self.get_latest_clone(clones).modified
        latest = [x for x in clones if x.modified == last]
        ip = request.META.get('HTTP_X_REAL_IP', '')
        ip_hash = int(sha1(ip).hexdigest(), 16)
        return latest[ip_hash % len(latest)]
