# coding: utf-8

from django.core.urlresolvers import reverse
from django.http import Http404

from rest_framework.response import Response

from intranet.dogma.dogma.api.framework import RepoAPIView
from intranet.dogma.dogma.api import serializers
from intranet.dogma.dogma.api.utils import make_absolute_url


class BranchView(RepoAPIView):
    """
    Бранч
    """

    serializer_class = serializers.DetailedBranchSerializer

    def get(self, request, source, owner, name, branch_name):
        branch = self.raw.get_branch(branch_name)

        if branch is None:
            raise Http404('Unknown branch: %s' % branch_name)

        commit = self.raw.branch_head(branch_name)

        commit_url = make_absolute_url(
            self.request,
            path=reverse('api:commit', args=[commit.hex])
        )

        data = {
            'name': branch.branch_name,
            'commit': {
                'url': commit_url,
                'sha': branch.target,
                'commit': {
                    'url': commit_url,
                    'author': {
                        'login': commit.author.login,
                        'name': commit.author.name,
                        'email': commit.author.email,
                        'date': commit.author.time,
                    },
                    'committer': {
                        'login': commit.committer.login,
                        'name': commit.committer.name,
                        'email': commit.committer.email,
                        'date': commit.committer.time,
                    },
                    'message': commit.message,
                    'tree': {
                        'url': make_absolute_url(
                            self.request,
                            path=reverse('api:tree', args=self.repo.args + [commit.tree.hex]),
                        ),
                        'sha': commit.tree.hex,
                    },
                },
                'parents': [
                    {
                        'url': make_absolute_url(
                            self.request,
                            path=reverse('api:commit', args=[parent.hex])
                        ),
                        'sha': parent.hex,
                    } for parent in commit.parents
                ],
            },
            '_links': {
                'self': make_absolute_url(
                    self.request,
                    path=reverse('api:branch', args=self.repo.args + [branch_name])
                ),
            }
        }

        return Response(self.get_serializer(data).data)


class BranchListView(RepoAPIView):
    """
    Список бранчей
    """

    serializer_class = serializers.BranchSerializer

    def get(self, request, source, owner, name):
        branches = []
        for branch_name in self.raw.list_branches():
            branch = self.raw.get_branch(branch_name)
            if branch:
                branches.append(branch)

        data = [{
            'name': branch.branch_name,
            'commit': {
                'sha': branch.target,
                'url': make_absolute_url(
                    request,
                    reverse('api:commit', args=[branch.target]),
                ),
            }
        } for branch in branches]

        return Response(self.get_serializer(data, many=True).data)
