# coding: utf-8
from datetime import datetime, timedelta
import pytz
import dateutil.parser as date_parser
from dateutil.relativedelta import relativedelta

from cached_property import cached_property
from django.core.urlresolvers import reverse
from django.utils.encoding import force_text
from django.core.paginator import Paginator, EmptyPage
from django.test import override_settings
from django.http import Http404

from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response
from rest_framework.exceptions import APIException
from rest_framework.generics import GenericAPIView

from intranet.dogma.dogma.api.framework import RepoAPIView, APIView
from intranet.dogma.dogma.api import serializers
from intranet.dogma.dogma.api.utils import to_positive_int, make_absolute_url
from intranet.dogma.dogma.api.logic.query import get_parsed_query, get_response_params, prepare_date
from intranet.dogma.dogma.core.logic.users import EmailGuesser
from intranet.dogma.dogma.core.logic.commits import commits_by_sha, get_stats_pattern, get_push_pattern
from intranet.dogma.dogma.core.dao.commits import get_commits_by_sha, get_commits_by_query, get_commits_stat_by_users_by_months, get_commits_count_by_day
from intranet.dogma.dogma.core.models import PushedCommit
from intranet.dogma.dogma.core.dao.users import get_users_id_by_login


class CommitView(RepoAPIView):
    """
    Коммит
    """
    serializer_class = serializers.DetailedCommitSerializer

    @cached_property
    def user_guesser(self):
        return EmailGuesser()

    def _guess_user(self, email, message=None):
        return self.user_guesser.guess_user_by(email, message)

    def _get_user_data(self, commit_user, message=None):
        """
        @rtype: dict
        """
        author, is_trusted_result = self._guess_user(commit_user.email, message)
        if not author:
            return {
                'login': commit_user.login,
                'name': commit_user.name,
                'email': commit_user.email,
                'date': commit_user.time,
                'uid': None,
                'is_trusted_login': False,
            }
        else:
            return {
                'login': author.login,
                'name': author.name,
                'email': author.email,
                'date': commit_user.time,
                'uid': author.uid,
                'is_trusted_login': is_trusted_result,
            }

    def get_commit_data(self, commit, detailed='short'):
        """
        @param detailed: short|media|full
        """
        commit_url = make_absolute_url(
            self.request,
            path=reverse('api:commit', args=[commit.hex]))

        id_ = self.model.backend.get_commit_native_id(self.raw, commit)

        data = {
            'url': commit_url,
            'sha': commit.hex,
            '_dogma': {
                'id': id_,
                'html_url': self.model.source.crawler.get_commit_url(self.model, id_),
            },
            'commit': {
                'url': commit_url,
                'author': self._get_user_data(commit.author, commit.message),
                'committer': self._get_user_data(commit.committer, commit.message),
                'message': commit.message,
                'tree': {
                    'url': make_absolute_url(
                        self.request,
                        path='not_implemented_for_hg',
                    ),
                    'sha': commit.tree.hex,
                },
            },
            'branch_name': [self.model.default_branch],
            'parents': [
                {
                    'url': make_absolute_url(
                        self.request,
                        path=reverse('api:commit', args=[parent.hex])),
                    'sha': parent.hex,
                } for parent in commit.parents
            ],
        }

        if detailed != 'short':
            diff = self.raw.commit_diff(commit)
            filename_to_patch = diff.detailed_patches()

            additions, deletions = diff.stats

            data.update(get_stats_pattern(additions=additions,
                                          deletions=deletions))
            if detailed == 'full':
                data['files'] = [
                    self.get_files_info(commit, patch, filename_to_patch)
                    for patch in diff.patches]

        return data

    def get(self, request, sha):
        try:
            commit = PushedCommit.objects.get(commit=sha)
        except PushedCommit.DoesNotExist:
            raise Http404('Unknown commit')

        commit_data = get_push_pattern(
            commit, detailed='medium',
            request=request,need_parents=False,
        )

        with override_settings(USE_TZ=False):
            return Response(self.get_serializer(commit_data).data)

    def get_files_info(self, commit, patch, filename_to_patch):
        deleted = patch.status == 'D'
        commit_sha = force_text(commit.hex if not deleted else commit.parents[0].hex)
        blob = self.raw.get_object_by_path(self.raw[commit_sha], patch.new_file_path)

        result = {
            'sha': patch.new_hex if not deleted else patch.old_hex,
            'status': patch.status_name,
            'filename': patch.new_file_path,
            'additions': patch.additions,
            'deletions': patch.deletions,
            'changes': patch.additions + patch.deletions,
            'raw_url': make_absolute_url(
                self.request,
                path=reverse('api:raw', args=self.repo.args + [commit_sha, patch.new_file_path])
            ),
            'blob_url': make_absolute_url(
                self.request,
                path=reverse('api:blob', args=self.repo.args + [blob.hex])
            ),

            'contents_url': make_absolute_url(
                self.request,
                path=reverse('api:contents', args=self.repo.args + [patch.new_file_path]),
                parameters=[('ref', commit_sha)]
            ),
            'patch': filename_to_patch.get(patch.new_file_path)
        }
        return result


class CommitListView(CommitView):
    """
    Список коммитов
    """

    serializer_class = serializers.DetailedCommitSerializer

    def get(self, request, source, owner, name):

        page_size = self.get_paginate_by()

        page = self.request.GET.get('page', 1)
        page = to_positive_int(page, 'page', strict_positive=True)
        sha = self.request.GET.get('sha')
        since = self.request.GET.get('since')

        mindate = datetime(1, 1, 1, 0, 0, tzinfo=pytz.UTC)
        if since is not None:
            since = date_parser.parse(since)
            if since.tzinfo is None or since.tzinfo.utcoffset(since) is None:
                since = since.replace(tzinfo=pytz.UTC)

        else:
            since = mindate

        pushed_commits = get_commits_by_sha(sha=sha, source=source,
                                            owner=owner, name=name,
                                            since=since)
        pushed_commits = list(pushed_commits)
        if since != mindate:
            pushed_commits.reverse()

        commits_to_skip = page_size * (page - 1)
        commits = []

        skipped = -1

        for commit in pushed_commits:
            if len(commits) >= page_size:
                break

            skipped += 1
            if skipped < commits_to_skip:
                continue
            else:
                commits.append(get_push_pattern(commit, detailed='medium', request=request))

        response = Response(self.get_serializer(commits, many=True).data)

        params = self.request.GET.dict()

        if commits and len(commits) >= page_size:
            params['page'] = page + 1
            response['Link'] = '<%s>; rel="next"' % (
                make_absolute_url(
                    request,
                    reverse('api:commits', args=self.repo.args),
                    parameters=list(params.items()),
                ),
            )
        return response


def path_in_commit(path, commit, walker):
    try:
        tree_entry = commit.tree[path]
    except KeyError:
        tree_entry = None

    parents = commit.parents

    if not parents:
        return tree_entry is not None

    # http://git-scm.com/docs/git-log
    num_treesame = 0

    for parent in parents:
        try:
            parent_entry = parent.tree[path]
        except KeyError:
            parent_entry = None

        if num_treesame > 0:
            walker.hide(parent.hex)
            continue

        if tree_entry is None and parent_entry is None:
            num_treesame += 1
        elif (tree_entry is not None and parent_entry is not None and
              tree_entry.hex == parent_entry.hex):
            num_treesame += 1

    if num_treesame == 0:
        return True
    else:
        return False


class CommitsSearchView(APIView):
    """
    Список коммитов, быстрая ручка, отдает
    данные из базы, не поднимая комиты с файловой
    системы, мимикрирует частично под ручку гитхаба:
    https://developer.github.com/v3/search/#search-commits

    Текущие поддерживаемые ручкой параметры являются ключами в AVAILABLE_QUERIES
    для добавления нового поля следует добавить запись следующего формата:
    (все ключи являются опциональными если добавить 'параметр_в_запросе':{}
    ручка будет всегда делать запрос вида
    Model.objects.filter(параметр_в_запросе__in=[значение_параметра])
    )

    'параметр_в_запросе' :{
        'field': 'author', указать если название параметра
                           не совпадает с названием поля модели
                           Если не указать 'field':
                           q=merge:true -> merge__in=['true']
                           Если указать 'field':'commit_type':
                           q=merge:true -> commit_type__in=['true']

        'query_suffix': 'overlap', указать если следует использовать
                                 оператор отличный от __in или
                                 если следует фильтровать по полю
                                 внешнего ключа

                                 Если не указать 'query_suffix':
                                 q=author:smosker -> author__in=['smosker']
                                 Если указать 'query_suffix': 'login__in':
                                 q=author:smosker -> author__login__in=['smosker']
                                 Если указать 'query_suffix': 'overlap':
                                 q=tickets:WIKI-1234 -> tickets_overlap=['WIKI-1234']

        'values_map' : {'true':'merge',
                        'false':'common',}, указать, если следует делать запрос в базу
                                            не по значению переданному в запросе, а по некому
                                            другому значению

                                            Если не указать 'values_map':
                                            q=some_value:true -> some_value__in=['merge']
                                            q=some_value:false -> some_value__in=['false']

                                            Если указать 'values_map': {'true':'merge'}:
                                            q=some_value:true -> some_value__in=['merge']
                                            q=some_value:false -> some_value__in=['false']

        'exclude': True, указать, если следует исключать по
                         указанному параметру

                         Если не указать 'exclude':
                         q=some_value:'test' -> .filter(some_value__in=['test'])
                         Если указать 'exclude': True:
                         q=some_value:'test' -> .exclude(some_value__in=['test'])

        'list': False, указать если значения нужно представлять
                       не в виде списка (поведение по умолчанию),
                       а в виде строки

                       Если не указать 'list':
                       q=some_value:'test' -> some_value__in=['test']
                       q=some_value:'test,test1' -> some_value__in=['test', 'test1']
                       Если указать 'list': False:
                       q=some_value:'test' -> some_value__in='test'
                       q=some_value:'test,test1' -> some_value__in='test test1'

        'apply_on_each': prepare_date, указать если к значениям параметра следует
                         применить какую либо функцию

                         Если не указать 'apply_on_each':
                         q=some_value:'2005-08-09T18:31:42 03, 2005-08-09T18:31:42 03'
                         -> .filter(some_value__in=['2005-08-09T18:31:42 03', '2005-08-09T18:31:42 03'])

                         Если указать 'apply_on_each': prepare_date:
                         q=some_value:'2005-08-09T18:31:42 03, 2005-08-09T18:31:42 03'
                         -> .filter(some_value__in=[prepare_date('2005-08-09T18:31:42 03'),
                                                    prepare_date('2005-08-09T18:31:42 03')])

        'apply_bulk': get_users_other_emails, указать если к следует применить функцию не к каждому
                                              значению параметра, а ко всему списку в целом

                                              Если не указать 'apply_bulk':
                                              q=some_value:'test,test1' -> .filter(some_value__in=['test', 'test1'])

                                              Если указать 'apply_bulk': get_users_other_emails:
                                              q=some_value:'test,test1'
                                              -> .filter(some_value__in=get_users_other_emails(['test', 'test1']))

                                              Должна возвращать iterable приемлемый для django orm

    }

    ADDITIONAL_PARAMETERS - прочие параметры, которые можно передавать
    в "q" запроса, но которые не влияют на выборку коммитов:

    'parents' - если передан данный параметр, то в ответ ручки
    по каждому коммиту будет включена информация о
    родительских коммитах

    """
    serializer_class = serializers.DetailedCommitSerializer
    AVAILABLE_QUERIES = {
        'author': {'query_suffix': 'id__in',
                   'apply_bulk': get_users_id_by_login,
                   },
        'committer': {'query_suffix': 'id__in',
                      'apply_bulk': get_users_id_by_login,
                      },
        'tickets': {'query_suffix': 'overlap', },
        'queues': {'query_suffix': 'overlap', },
        'author-email': {'field': 'author',
                         'query_suffix': 'email__in',
                         },
        'committer-email': {'field': 'committer',
                            'query_suffix': 'email__in',
                            },
        'author-uid': {'field': 'author',
                       'query_suffix': 'uid__in',
                       },
        'committer-uid': {'field': 'committer',
                          'query_suffix': 'uid__in',
                          },
        'merge': {'field': 'commit_type',
                  'values_map': {
                      'true': PushedCommit.TYPES.merge,
                      'false': PushedCommit.TYPES.common,
                      },
                  },
        'exclude_dublicates': {
            'field': 'message',
            'query_suffix': 'contains',
            'exclude': True,
            'list': False,
            'values_map': {
                'true': 'git-svn-id',
            },
        },
        'source': {
            'field': 'repo__source',
            'query_suffix': 'code__in',
        },
        'commit_time': {
            'query_suffix': 'range',
            'apply_on_each': prepare_date
        },
    }
    ADDITIONAL_PARAMETERS = {'parents', }

    def get(self, request):
        page_size = self.get_paginate_by()

        page = self.request.GET.get('page', 1)
        page = to_positive_int(page, 'page', strict_positive=True)
        q = self.request.GET.get('q')
        need_num_pages = self.request.GET.get('num_pages')

        include_query, exclude_query, additional_parameters = self.get_params_by_q(q)
        commits = get_commits_by_query(include_query, exclude_query)
        commits = commits.order_by('-commit_time')
        need_parents = 'parents' in additional_parameters
        if need_parents:
            commits = commits.prefetch_related('parents')

        paginator = Paginator(commits, page_size)

        try:
            commits = paginator.page(page)
        except EmptyPage:
            commits = []

        commits_response = []

        for commit in commits:
                commits_response.append(get_push_pattern(commit,
                                                         detailed='medium',
                                                         request=request,
                                                         need_parents=need_parents,
                                                         )
                                        )
        with override_settings(USE_TZ=False):
            response = Response(self.get_serializer(commits_response, many=True).data)

        params = self.request.GET.dict()

        if need_num_pages:
            response['num_pages'] = paginator.num_pages

        if commits and commits.has_next():
            params['page'] = commits.next_page_number()
            response['Link'] = '<%s>; rel="next"' % (
                make_absolute_url(
                    request,
                    reverse('api:commits_search'),
                    parameters=list(params.items()),
                ),
            )
        return response

    def get_params_by_q(self, q):
        if not q:
            error = 'You have to specify at least one search parameter'
            raise APIException(error)

        parsed_query = get_parsed_query(q)
        include_query, exclude_query, additional_parameters = get_response_params(parsed_query,
                                                                                  self.AVAILABLE_QUERIES,
                                                                                  self.ADDITIONAL_PARAMETERS,
                                                                                  )
        if not (include_query or exclude_query):
            error = 'Request with incorrect parameters was made'
            raise APIException(error)

        return include_query, exclude_query, additional_parameters


class CommitsSearchCountView(CommitsSearchView):
    renderer_classes = (JSONRenderer,)

    def get(self, request):
        q = self.request.GET.get('q')
        include_query, exclude_query, additional_parameters= self.get_params_by_q(q)

        commits = get_commits_by_query(include_query, exclude_query)
        result = {'commits_count': commits.count()}

        return Response(result)


class CommitsStatByMonthsView(GenericAPIView):
    renderer_classes = (JSONRenderer,)

    def get(self, request):
        logins = logins = request.GET.get("logins") or request.GET.get("login")
        logins = logins.split(",")
        if len(logins) > 20:
            raise APIException("Too many logins, max: 20")

        data = get_commits_stat_by_users_by_months(logins)

        result = dict()

        for login, commits_count, lines_added, lines_deleted, month, year in data:
            if login not in result:
                result[login] = []
            result[login].append({
                "year": int(year),
                "month": int(month),
                "commits_count": int(commits_count),
                "lines_added": int(lines_added),
                "lines_deleted": int(lines_deleted),
            })

        return Response(result)


class CommitsHeatmap(GenericAPIView):
    renderer_classes = (JSONRenderer,)

    def get(self, request):
        login = request.GET.get("login")
        start = request.GET.get("commit_time_from")
        end = request.GET.get("commit_time_to")

        if not login:
            raise APIException("No login")

        def _parse_date(string):
            date = date_parser.parse(string)
            if date.tzinfo is None or date.tzinfo.utcoffset(date) is None:
                date = date.replace(tzinfo=pytz.UTC)

            offset = date.tzinfo.utcoffset(date)

            return date, offset

        offset = timedelta(0)

        if not start:
            start = datetime.now() - relativedelta(months=1)
        else:
            start, offset = _parse_date(start)

        if not end:
            end = datetime.now()
        else:
            end, offset = _parse_date(end)

        if start > end - relativedelta(days=1):
            raise APIException("Interval must be larger than one day")

        data = get_commits_count_by_day(login, start, end, offset)
        by_day = dict()

        for _, commits_count, day in data:
            by_day[day.strftime("%Y-%m-%d")] = commits_count

        res = []

        date_iter = start
        while date_iter < end:
            date_string = date_iter.strftime("%Y-%m-%d")
            res.append({
                "date": date_string,
                "count": by_day.get(date_string, 0)
            })
            date_iter += relativedelta(days=1)

        return Response({"data": res})
