# coding: utf-8

from django.http import Http404

from rest_framework.response import Response
from rest_framework.reverse import reverse

from intranet.dogma.dogma.api.utils import make_absolute_url

from .commit import CommitView


class CompareView(CommitView):
    """
    Сравнение двух ревизий
    https://developer.github.com/v3/repos/commits/#compare-two-commits
    """
    def get(self, request, source, owner, name, old_branch_or_tag_or_sha, new_branch_or_tag_or_sha):
        repo = self.raw

        try:
            old = repo.get_object_by_refspec(old_branch_or_tag_or_sha)
            new = repo.get_object_by_refspec(new_branch_or_tag_or_sha)
        except ValueError:
            raise Http404("Not found")

        def patch_info(patch):
            return {
                'sha': patch.new_hex,
                'filename': patch.new_file_path,
                'additions': patch.additions,
                'deletions': patch.deletions,
                'changes': patch.additions + patch.deletions,
                'status': patch.status_name,
                'contents_url': make_absolute_url(
                    request,
                    reverse('api:contents',
                            args=self.repo.args + [patch.new_file_path]),
                    [('ref', new.hex)],
                ),
                'raw_url': make_absolute_url(
                    request,
                    reverse('api:raw',
                            args=self.repo.args + [new.hex, patch.new_file_path]),
                )
            }

        # Получаем initial commit первого бранча
        init_old = next(iter(repo.walk(old.hex, 'tp+rv')))
        init_new = next(iter(repo.walk(new.hex, 'tp+rv')))

        params = (init_old.hex, repo.path)
        assert not init_old.parents, "%s not initial commit in %s" % params
        assert not init_new.parents, "%s not initial commit in %s" % params

        if init_old.hex != init_new.hex:
            raise Http404("%s and %s are entirely different commit histories." %
                          (old_branch_or_tag_or_sha, new_branch_or_tag_or_sha))

        if old.hex == new.hex:
            commits = []
        else:
            raw_commits = list(repo.commits_between(old, new))
            commits = [self.get_commit_data(cm, detailed='short')
                       for cm in raw_commits]

        if commits:
            parents = raw_commits[-1].parents
            against = parents[0] if parents else -1
            diff = repo.commit_diff(new, against=against)
            files = list(map(patch_info, diff.patches))
        else:
            files = []

        old_info = self.get_commit_data(old, detailed='short')

        data = {
            'url': make_absolute_url(
                request,
                reverse('api:compare', args=self.repo.args +
                                            [old_branch_or_tag_or_sha, new_branch_or_tag_or_sha]),
            ),
            'base_commit': old_info,
            'files': files,
            'commits': commits,
            'total_commits': len(commits),
        }

        return Response(data)
