# coding: utf-8
import os
import mimetypes

from django.http import HttpResponse, Http404
from django.core.urlresolvers import reverse
from django.utils.encoding import force_text

from rest_framework.response import Response


from intranet.dogma.dogma.core.git.models import Commit, Blob
from intranet.dogma.dogma.api.framework import RepoAPIView
from intranet.dogma.dogma.api.utils import make_absolute_url
from intranet.dogma.dogma.api import serializers


class ContentView(RepoAPIView):
    """
    Контент
    """

    def get(self, request, source, owner, name, path):
        refspec = request.GET.get('ref') or force_text(self.raw.head.target)

        try:
            commit = self.raw.get_object_by_refspec(refspec)
            entries = self.raw.get_tree_entries(commit, path)
            data = get_contents(request, self.repo, refspec, entries)
        except (KeyError, ValueError, RuntimeError) as exc:
            raise Http404(exc.message)

        if isinstance(data, list):
            serialized_data = serializers.ContentSerializer(data, many=True).data
        else:
            if data['type'] == 'submodule':
                serialized_data = serializers.SubmoduleContentSerializer(data).data
            else:
                serialized_data = serializers.DetailedContentSerializer(data).data

        # TODO: По идее тут еще могут быть симлинки

        return Response(serialized_data)


class RawView(RepoAPIView):
    """
    Исходный файл
    """

    def get(self, request, source, owner, name, sha, path):
        try:
            obj = self.repo.raw.get_object_by_refspec(sha)
            if isinstance(obj, Commit):
                obj = self.repo.raw.get_object_by_path(obj, path)
            elif isinstance(obj, Blob):
                pass
            else:
                raise ValueError("Expected Blob or Commit for %s, not %s" %
                                 (sha, type(obj)))

        except (ValueError, KeyError) as exc:
            raise Http404(exc.message)

        mimetype = mimetypes.guess_type(path)[0] or 'text/plain'
        return HttpResponse(obj.data, content_type=mimetype)


def get_contents(request, repo, refspec, entries):
    def inner(item):
        type_, path, obj = item

        contents_url = make_absolute_url(
            request,
            reverse('api:contents', args=repo.args + [path]),
            [('ref', refspec)],
        )
        git_url = make_absolute_url(
            request,
            reverse('api:blob', args=repo.args + [obj.hex])
        )

        result = {
            'type': type_,
            'sha': obj.hex,
            'name': os.path.basename(path),
            'path': path,
            'url': contents_url,
            'git_url': git_url,
            'size': getattr(obj, 'size', 0),
            'is_binary': getattr(obj, 'is_binary', False),
        }

        if isinstance(obj, Blob):
            if (not obj.is_binary) or obj.size <= 1024 ** 2:
                encoding, data = obj.encode_data()
                result['encoding'] = encoding
                result['content'] = data
            else:
                result['encoding'] = None
                result['content'] = None

            result['download_url'] = make_absolute_url(
                request,
                reverse('api:raw', args=repo.args + [obj.hex, path]),
            )

        return result

    if isinstance(entries, list):
        return list(map(inner, entries))
    else:
        return inner(entries)
