# coding: utf-8

from django.db.models import Count
from django.conf import settings

from rest_framework.mixins import RetrieveModelMixin

from intranet.dogma.dogma.api.framework import APIView, RepoAPIView, ListModelMixin
from intranet.dogma.dogma.api import serializers
from intranet.dogma.dogma.api.utils import to_positive_int
from intranet.dogma.dogma.api.permissions import OnlyTrustedUsersPermission
from intranet.dogma.dogma.core.models import Repo
from intranet.dogma.dogma.core.logic.users import is_trusted_user
from intranet.dogma.dogma.core.utils import get_current_node


class RepositoryView(RetrieveModelMixin, RepoAPIView):
    """
    Информация о репозитории
    """

    serializer_class = serializers.RepositorySerializer
    permission_classes = OnlyTrustedUsersPermission,

    def get_object(self):
        return self.model

    def get(self, request, source, owner, name):
        return self.retrieve(request, source, owner, name)


class RepositoryListMixin(ListModelMixin):
    def paginate_queryset(self, queryset):
        """
        Pagination для списка репозиториев сделана через since
        """
        page_size = self.get_paginate_by()

        if settings.DOGMA_API_DEFAULT_PAGINATE_BY != page_size:
            links = [('first', [('per_page', page_size)])]
        else:
            links = [('first', [])]

        # Поддерживаем паджинацию и по page, и по since
        if 'page' in self.request.GET:
            page = self.request.GET['page']
            page = to_positive_int(page, 'page', strict_positive=True)

            total_count = queryset.count()
            pages_count = int(total_count / page_size) + (1 if total_count % page_size else 0)

            queryset = queryset.order_by('id')[((page - 1) * page_size):(page * page_size)]

            links.append(('last', [('page', pages_count), ('per_page', page_size)]))

        else:
            since = self.request.GET.get('since', 0)
            since = to_positive_int(since, 'since', strict_positive=False)

            queryset = queryset.filter(id__gt=since)[:page_size]

            if queryset:
                new_since = max(o.id for o in queryset)
                links.append(('next', [('since', new_since), ('per_page', page_size)]))

        return queryset, links


class RepositoryListView(RepositoryListMixin, APIView):
    """
    Список всех репозиториев.
    """

    serializer_class = serializers.RepositorySerializer

    def get_queryset(self):
        qs = (
            Repo.objects
            .filter(clones__status='active')
            .annotate(clone_count=Count('clones'))
            .filter(clone_count__gt=0, source__hidden=False)
            .order_by('id')
            .select_related('source')
        )

        if not is_trusted_user(self.request.yauser):
            qs = qs.filter(is_public=True)

        if not self.request.is_global:
            qs = qs.filter(clones__node=get_current_node())

        return qs

    def get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class UserRepositoryListView(RepositoryListView):
    """
    Список репозиториев пользователя.
    """

    def get_queryset(self):
        queryset = super(UserRepositoryListView, self).get_queryset()
        return queryset.filter(
            source__code=self.kwargs['source'],
            owner=self.kwargs['owner'],
            source__hidden=False,
        )
