# coding: utf-8
import logging

from urllib.parse import urlparse

from rest_framework.renderers import JSONRenderer
from rest_framework import filters
from dir_data_sync.org_ctx import get_org

from intranet.dogma.dogma.api.framework import APIView
from intranet.dogma.dogma.core.dao.source import get_source_by_host
from intranet.dogma.dogma.api.errors import BadRequestError
from intranet.dogma.dogma.api.pagination import DogmaLimitOffsetPagination
from intranet.dogma.dogma.core.models import Credential

log = logging.getLogger(__name__)


class BaseV4ApiView(APIView):
    pagination_class = DogmaLimitOffsetPagination
    filter_backends = filters.OrderingFilter,
    renderer_classes = JSONRenderer,

    def get_source(self, host):
        source = get_source_by_host(host)
        if not source:
            # Пока работаем только с созданными руками источниками
            log.error('No source found "%s"', host)
            error = 'No source with such host found'
            raise BadRequestError(error)
        return source

    def parse_url(self, url):
        parsed_url = urlparse(url)
        path = parsed_url.path.strip('/')
        host = parsed_url.netloc

        if not all((host, path)):
            log.warning('Cant parse url "%s"', url)
            error = 'Request with incorrect url was made'
            raise BadRequestError(error)

        return path, host.split('@')[-1]

    def get_request_data(self):
        url = self.request.data.get('url')
        credentials = self.request.data.get('credentials')
        if not url:
            error = 'You have to provide url'
            raise BadRequestError(error)
        return url, credentials

    def add_connections(self, repo, credentials):
        repo.connect_organization.add(get_org())
        if credentials:
            repo.credentials.add(*credentials)

    def change_credentials_for_obj(self, obj):
        credentials = self.request.data.get('credentials', False)
        if credentials is False:
            error = 'You have to provide credentials'
            raise BadRequestError(error)
        credentials_objects = (
            Credential.objects
            .get_objects_for_current_org()
            .filter(pk__in=credentials)
        )
        if len(credentials_objects) != len(credentials):
            error = 'No credentials with such ids'
            raise BadRequestError(error)

        obj.credentials.set(credentials_objects)
