# coding: utf-8
import logging
import re

from dir_data_sync.org_ctx import get_org

from rest_framework import mixins
from rest_framework import viewsets
from rest_framework.decorators import action
from rest_framework.response import Response

from ylog.context import log_context

from django.db import transaction

from intranet.dogma.dogma.core.models import OrganisationsToClone
from intranet.dogma.dogma.core.dao.repo import disable_repos
from intranet.dogma.dogma.core.dao.organization import get_or_create_organization
from intranet.dogma.dogma.api.errors import BadRequestError

from .base import BaseV4ApiView
from ..serializers import OrganizationSerializer

log = logging.getLogger(__name__)


class OrganizationView(BaseV4ApiView,
                       mixins.RetrieveModelMixin,
                       mixins.ListModelMixin,
                       viewsets.GenericViewSet,
                       ):
    model = OrganisationsToClone
    serializer_class = OrganizationSerializer
    ordering = 'source_id', 'name', 'id',

    def get_queryset(self):
        return (self.model.objects
                .get_objects_for_current_org()
                .select_related('source')
                .prefetch_related('credentials')
                )

    def prepare_name(self, org_name, source):
        """
        Для битбакета отдельная логика, у него ссылка на
        проект выглядят как
        https://bitbucket.org/account/user/dogmatestb2b/projects/NEWTEST
        """
        if 'bitbucket.org' in source.web_url:
            parsed_name = re.search(
                r'account/user/(?P<team>.+?)/projects/(?P<project>.+)$',
                org_name
            )
            if not parsed_name:
                error = 'Bad url passed'
                raise BadRequestError(error)
            team = parsed_name.group('team')
            project = parsed_name.group('project')
            org_name = '{}-{}'.format(team, project)

        return org_name

    def create(self, request, *args, **kwargs):
        url, credentials = self.get_request_data()
        org_name, host = self.parse_url(url)
        with log_context(org_name=org_name, host=host):
            source = self.get_source(host)
            org_name = self.prepare_name(org_name, source)
            if '/' in org_name:
                error = 'You should pass organization url, not repository'
                raise BadRequestError(error)
            with transaction.atomic():
                organization, created = get_or_create_organization(org_name=org_name, source=source)
                if not created:
                    organization.make_active()

                self.add_connections(organization, credentials)

            serializer = self.get_serializer(organization)
            return Response(serializer.data)

    @action(methods=['post'], detail=True, url_path='disable', url_name='disable')
    def disable_organization(self, request, *args, **kwargs):
        with transaction.atomic():
            org = self.get_object()
            repos = org.repo.all()
            disable_repos(*repos)
            org.connect_organization.remove(get_org())
            if not org.connect_organization.exists():
                org.is_active = False
                org.save()
        return Response(data={'status': 'ok'})

    @action(methods=['post'], detail=True, url_path='change-credentials', url_name='change-credentials')
    def change_credentials(self, request, *args, **kwargs):
        obj = self.get_object()
        self.change_credentials_for_obj(obj)
        serializer = self.get_serializer(obj)
        return Response(serializer.data)

    def change_credentials_for_obj(self, obj):
        response = super(OrganizationView, self).change_credentials_for_obj(obj)
        for repo in obj.repo.prefetch_related('clones'):
            repo.after_credentials_change()
        return response
