# coding: utf-8
import logging

from django.db import transaction
from django.conf import settings

from rest_framework import mixins
from rest_framework import viewsets
from rest_framework.decorators import action
from rest_framework.response import Response

from ylog.context import log_context

from intranet.dogma.dogma.core.models import Repo
from intranet.dogma.dogma.core.dao.repo import get_or_create_repo, disable_repos
from intranet.dogma.dogma.core.logic.repos import make_valid_url
from intranet.dogma.dogma.core.tasks import clone_repo
from intranet.dogma.dogma.core.utils import get_random_node_queue
from intranet.dogma.dogma.api.errors import BadRequestError

from .base import BaseV4ApiView
from ..serializers import RepoSerializer

log = logging.getLogger(__name__)


class RepoView(BaseV4ApiView,
               mixins.RetrieveModelMixin,
               mixins.ListModelMixin,
               viewsets.GenericViewSet,
               ):
    model = Repo
    serializer_class = RepoSerializer
    ordering = 'source_id', 'owner', 'name', 'id',

    def get_queryset(self):
        return (self.model.objects
                .get_objects_for_current_org()
                .select_related('source', 'organisation', 'organisation__source',)
                .prefetch_related('credentials', 'organisation__credentials',)
                )

    def create(self, request, *args, **kwargs):
        url, credentials = self.get_request_data()
        vcs_name, host = self.parse_url(url)
        if host != settings.GITLAB_HOST:
            vcs_name = '/'.join(vcs_name.split('/')[:2])
        if vcs_name.endswith('.git'):
            vcs_name = vcs_name[:-4]

        with log_context(vcs_name=vcs_name, host=host):
            source = self.get_source(host)
            try:
                owner, name = vcs_name.split('/', 1)
                url = make_valid_url(source=source, owner=owner, name=name)
            except ValueError:
                error = 'Request with incorrect url was made'
                raise BadRequestError(error)
            with transaction.atomic():
                repo, created = get_or_create_repo(
                    vcs_name=vcs_name, name=name,
                    owner=owner, url=url, source=source,
                )
                if not repo.is_active:
                    repo.is_active = True
                    repo.save()

                self.add_connections(repo, credentials)

            if created:
                log.info('Sending task for cloning new repo')
                clone_repo.apply_async(
                    args=[repo.id],
                    queue=get_random_node_queue('clone'),
                )

            serializer = self.get_serializer(repo)
            return Response(serializer.data)

    @action(methods=['post'], detail=True, url_path='disable', url_name='disable')
    def disable_repo(self, request, *args, **kwargs):
        obj = self.get_object()
        disable_repos(obj)
        return Response(data={'status': 'ok'})

    @action(methods=['post'], detail=True, url_path='change-credentials', url_name='change-credentials')
    def change_credentials(self, request, *args, **kwargs):
        obj = self.get_object()
        self.change_credentials_for_obj(obj)
        serializer = self.get_serializer(obj)
        return Response(serializer.data)

    def change_credentials_for_obj(self, obj):
        response = super(RepoView, self).change_credentials_for_obj(obj)
        obj.after_credentials_change()
        return response
