# coding: utf-8
"""
Абстракция над Git и Mercurial.
"""



import os

from unidiff import PatchSet
from schematics.models import Model
from schematics.types import (StringType, DateTimeType, BooleanType,
                              SHA1Type, IntType, EmailType)
from schematics.types.compound import ListType, ModelType


class Base(Model):
    def _convert(self, raw, **kwargs):
        fields = kwargs.get('fields', list(self._fields.keys()))
        return {name: getattr(raw, name) for name in fields}

    def __init__(self, raw_data, *args, **kwargs):
        self._initial = raw_data
        super(Base, self).__init__(raw_data, *args, **kwargs)


class Repository(Base):
    path = StringType()
    workdir = StringType()
    is_bare = BooleanType()
    is_empty = BooleanType()

    @classmethod
    def discover(cls, path):
        raise NotImplementedError

    def __getitem__(self, hex_):
        res = self.get(hex_)
        if res is None:
            raise KeyError(hex_)
        return res

    def walk(self, hex, sort_type=''):
        """ sort_type:
          'tp' - topological,
          'tm' - time,
          'tp+rv' - topological reverse,
          'tm+rv' - time reverse
        """
        raise NotImplementedError

    def all_commits(self, exclude):
        """
        Вернуть все комиты всего репозитория. Комиты берутся из каждого бранча по очереди.
        Если какой-то коммит встречается в двух бранчах, то он возвращается один раз.
        Частично упорядочены от новых к старым.

        @rtype: list
        """
        raise NotImplementedError


class GitObject(Base):
    hex = SHA1Type()


class TreeEntry(GitObject):
    name = StringType()
    filemode = IntType()


class Tree(GitObject):
    def __iter__(self):
        for i in self._initial:
            yield TreeEntry(i)

    def __getitem__(self, name):
        return TreeEntry(self._initial[name])


class Branch(Base):
    branch_name = StringType()
    name = StringType()
    target = SHA1Type()


class Diff(Base):

    @property
    def patches(self):
        """
        @rtype: list
        @return: list of Patches
        """
        raise NotImplementedError

    def detailed_patches(self):
        """
        @rtype: dict
        @return: additions, deletions
        """
        raise NotImplementedError

    @property
    def stats(self):
        """
        @rtype: list
        @return: list of 2 int: additions, deletions
        """
        patches = self.patches
        additions = sum(patch.additions for patch in patches)
        deletions = sum(patch.deletions for patch in patches)
        return additions, deletions


class Patch(Base):
    GIT_STATUS_TO_NAME = {
        'modified': 'M',
        'added': 'A',
        'renamed': 'R',
        'removed': 'D',
    }

    old_file_path = StringType()
    new_file_path = StringType()
    old_hex = SHA1Type()
    new_hex = SHA1Type()
    status = StringType()
    status_name = StringType()
    additions = IntType()
    deletions = IntType()
    is_binary = BooleanType()


class User(Base):
    login = StringType()
    uid = StringType()
    name = StringType()
    email = EmailType()
    time = DateTimeType()


class Commit(GitObject):
    author = ModelType(User)
    committer = ModelType(User)
    message = StringType()
    parent_ids = ListType(SHA1Type())
    commit_time = DateTimeType()


class Walker(object):
    _initial = None

    def __init__(self, initial):
        self._initial = initial

    def __iter__(self):
        raise NotImplementedError

    def hide(self, hex):
        raise NotImplementedError


class ChangedFilesMixin(object):

    def get_patch_data(self):
        """
        @rtype: basestring
        """
        return NotImplemented

    def get_patch_set(self):
        patch_data = self.get_patch_data()
        if patch_data:
            return PatchSet(patch_data)

    @property
    def changed_files(self):
        changed_files_data = {}
        patches = self.get_patch_set()
        if patches:
            for patch in patches:
                file_name = self.get_file_name(patch)
                extension = self.get_file_extension(file_name)
                changed_files_data[file_name] = {'additions': patch.added,
                                                 'deletions': patch.removed,
                                                 'extension': extension,
                                                 }
        return changed_files_data

    def get_file_extension(self, file_name):
        if isinstance(file_name, str):
            if file_name.endswith('changelog'):
                return 'changelog'
            filename, file_extension = os.path.splitext(file_name)
            if file_extension:
                return file_extension[1:]

        return 'unknown extension'

    def get_file_name(self, patch):
        file_name = patch.source_file
        if file_name == '/dev/null':
            file_name = patch.target_file
        return file_name
