# coding: utf-8


from django.contrib import admin
from django.conf import settings

from . import models


class NodeAdmin(admin.ModelAdmin):
    list_display = 'name', 'hostname', 'space_total', 'space_available', 'enabled'
    search_fields = list_display
    ordering = '-space_available',


class SourceAdmin(admin.ModelAdmin):
    list_display = 'name', 'code', 'vcs_type', 'web_type', 'web_url', 'host', 'rate', 'ratio'
    search_fields = list_display
    readonly_fields = ('last_sync_success_time', 'last_sync_fail_time',
                       'last_sync_fail_trace', 'error_value',
                       'last_sync_fail_error_code',
                       )


class RepoAdmin(admin.ModelAdmin):
    list_select_related = 'source',

    list_display = 'name', 'owner', 'vcs_name', 'source'
    list_filter = 'source',
    search_fields = 'name', 'owner', 'vcs_name'
    ordering = 'name',
    raw_id_fields = 'parent',
    readonly_fields = ('last_sync_success_time',
                       'last_sync_fail_time', 'last_sync_fail_trace',
                       'contiguous_chain_of_commits_ends_at',
                       'created', 'modified', 'last_sync_fail_error_code',
                       )


class CloneAdmin(admin.ModelAdmin):
    list_select_related = 'repo', 'node',

    list_display = 'repo', 'node', 'status', 'space_required', 'path'
    list_filter = 'node', 'status'
    search_fields = 'status', 'space_required', 'path'
    raw_id_fields = 'repo',


class BranchStampInline(admin.StackedInline):
    model = models.BranchStamp
    can_delete = False
    extra = 0


class FetchEventAdmin(admin.ModelAdmin):
    list_select_related = 'clone',
    list_display = 'clone', 'created'
    inlines = [BranchStampInline]


class BranchStampAdmin(admin.ModelAdmin):
    list_display = 'fetch_event', 'name', 'head', 'head_time', 'head_message'
    search_fields = 'name', 'head', 'head_message'
    ordering = 'fetch_event',


class OrgsToCloneAdmin(admin.ModelAdmin):
    list_select_related = 'source',
    list_display = 'source', 'name'
    search_fields = 'name',


def get_slave_or_master():
    databases = settings.DATABASES.copy()
    del databases['default']
    if databases:
        return next(iter(databases.keys()))
    return 'default'


class PushedCommitAdmin(admin.ModelAdmin):
    read_from_db = get_slave_or_master()
    list_per_page = 20

    list_display = 'commit', 'commit_time', 'created',
    raw_id_fields = 'repo', 'author', 'committer', 'parents',
    search_fields = 'repo__name',
    readonly_fields = 'commit', 'repo', 'created', 'modified',

    def get_queryset(self, request):
        # Tell Django to look for objects on the 'other' database.
        return super(PushedCommitAdmin, self).get_queryset(request).using(self.read_from_db)

class PushedCommitDuplicateHexAdmin(admin.ModelAdmin):
    read_from_db = get_slave_or_master()
    list_per_page = 20

    list_display = 'commit',
    raw_id_fields = 'repo',
    search_fields = 'repo__name',
    readonly_fields = 'commit', 'repo',

    def get_queryset(self, request):
        # Tell Django to look for objects on the 'other' database.
        return super(PushedCommitDuplicateHexAdmin, self).get_queryset(request).using(self.read_from_db)


class CredentialAdmin(admin.ModelAdmin):
    list_display = 'name', 'is_success', 'connect_organization',


admin.site.register(models.Node, NodeAdmin)
admin.site.register(models.Source, SourceAdmin)
admin.site.register(models.Repo, RepoAdmin)
admin.site.register(models.Clone, CloneAdmin)
admin.site.register(models.FetchEvent, FetchEventAdmin)
admin.site.register(models.BranchStamp, BranchStampAdmin)
admin.site.register(models.OrganisationsToClone, OrgsToCloneAdmin)
admin.site.register(models.PushedCommit, PushedCommitAdmin)
admin.site.register(models.PushedCommitDuplicate, PushedCommitAdmin)
admin.site.register(models.Credential, CredentialAdmin)
admin.site.register(models.PushedCommitDuplicateHex, PushedCommitDuplicateHexAdmin)
