# coding: utf-8



import logging
from os import path

import grpc
from grpc._channel import _MultiThreadedRendezvous

from django.conf import settings
import chardet
from cached_property import cached_property
from schematics.types import StringType, DateTimeType, SHA1Type
from schematics.types.compound import ListType, ModelType
from pretend import stub
from intranet.dogma.dogma.core.git.models import dtfromts
from intranet.dogma.dogma.core.hg.models import to_unicode

from arc.api.public import repo_pb2_grpc, repo_pb2, shared_pb2
from .. import abstract_repository

log = logging.getLogger(__name__)


class Repository(abstract_repository.Repository):

    @classmethod
    def discover(cls, path):
        return cls(path)

    def __init__(self, *args, **kwargs):

        super(Repository, self).__init__(*args, **kwargs)

        clone = self._initial
        self.branch = clone.repo.vcs_name
        self.source = clone.repo.source

        if self.branch == "trunk":
            if clone.repo.contiguous_chain_of_commits_ends_at is None:
                if self.branch == "trunk":
                    self.last_commit = self.source.extra_info.get('start_from')  # 'r7870219'
            else:
                self.last_commit = clone.repo.contiguous_chain_of_commits_ends_at.commit_id
        else:
            self.last_commit = "trunk"

        call_credentials = grpc.access_token_call_credentials(self.source.extra_info.get('token'))
        composite_credentials = grpc.composite_channel_credentials(
            grpc.ssl_channel_credentials(),
            call_credentials
        )

        public_api = grpc.secure_channel(
            target=self.source.extra_info.get('endpoint'),
            credentials=composite_credentials
        )

        self.history_service = repo_pb2_grpc.HistoryServiceStub(public_api)
        self.diff_service = repo_pb2_grpc.DiffServiceStub(public_api)
        self.branch_service = repo_pb2_grpc.BranchServiceStub(public_api)

        self._limit = self.source.extra_info.get('limit')
        self._filter = self.source.extra_info.get('filter')

    def _convert(self, raw, **kwargs):
        info = super(Repository, self)._convert(raw, fields=('path',), **kwargs)
        info.update({
            'workdir': "",
            'is_bare': False,
            'is_empty': False,
        })
        return info

    def list_branches(self, *args, **kwargs):
        if self.branch == "trunk":
            return ["trunk"]
        else:
            return self.discover_branches()

    def discover_branches(self):
        req = repo_pb2.ListRefsRequest()
        req.Inconsistent = True
        req.Lightweight = True
        req.PrefixFilter = self.branch + "/"
        res = []
        for item in self.branch_service.ListRefs(req):
            for data in item.Refs:
                res.append(data.Name)
        return res

    def commit_diff(self, commit, against=None):
        commit = commit._initial["commit"]

        req = repo_pb2.DiffstatRequest()
        req.ToRevision = commit.Oid
        req.Mode = shared_pb2.FlatPath

        try:
            diff_data = []
            try:
                for chunk in self.diff_service.Diffstat(req):
                    print(chunk)
                    for f in chunk.Files:
                        diff_data.append(f)
            except Exception:
                log.exception("Can not get diff data for commit " + str(commit))
        except _MultiThreadedRendezvous as e:
            if not "cannot read encrypted file" in e.details():  # XXX
                raise
        return Diff(commit, diff_data)

    def walk(self, *args, **kwargs):
        return super(Repository, self).walk(*args, **kwargs)

    def commits_between(self, cm1, cm2):
        raise NotImplementedError

    def lookup_note(self, *args, **kwargs):
        raise NotImplementedError

    @property
    def head(self):
        raise ValueError

    def all_commits(self, exclude):
        for branch in self.list_branches():
            request = repo_pb2.LogRequest(
                StartRevisions=[branch],
                HideRevisions=[self.last_commit],
                Path="/",
            )

            c = 0
            print("Request: {}".format(request))

            commits = []

            for resp in self.history_service.Log(request):
                for raw_commit in resp.Commits:
                    print(raw_commit)
                    if not self._filter or raw_commit.Commit.Author == self._filter:
                        if raw_commit.Commit.Oid not in exclude:
                            commits.append(Commit({
                                "commit": raw_commit.Commit,
                                "branch": branch,
                            }))
                    c += 1
                    if self._limit and c > self._limit:
                        break
                if self._limit and c > self._limit:
                    break

            for commit in commits[::-1]:
                yield commit


class User(abstract_repository.User):

    def _convert(self, raw, **kwargs):

        email = raw + "@yandex-team.ru"
        name = raw
        login = raw

        info = {
            'login': login,
            'time': None,
            'uid': None,
            'email': email,
            'name': name,
        }
        return info


class Tree(abstract_repository.GitObject):
    def _convert(self, raw, **kwargs):
        return {
            'hex': raw
        }

    def __iter__(self):
        raise NotImplementedError

    def __getitem__(self, name):
        raise NotImplementedError


class Commit(abstract_repository.GitObject):
    author = ModelType(User)
    committer = ModelType(User)
    message = StringType()
    tree = ModelType(Tree)
    parent_ids = ListType(SHA1Type())
    commit_time = DateTimeType()
    branch_name = StringType()

    @property
    def parents(self):
        return []

    @property
    def merge_commit(self):
        return False

    def _convert(self, raw, **kwargs):
        commit = raw["commit"]
        branch = raw["branch"]
        user = User(commit.Author)
        data = {
            'hex': commit.Oid,
            'author': user,
            'committer': user,
            'message': to_unicode(commit.Message),
            'parent_ids': [p for p in commit.ParentOids],
            'commit_time': dtfromts(commit.Timestamp.seconds),
            'tree': Tree(commit.TreeOid),
            'branch_name': branch,
        }
        return data

    def __repr__(self):
        return '<Commit: %s>' % self.hex[:8]

    __str__ = __repr__


class Diff(abstract_repository.ChangedFilesMixin, abstract_repository.Diff):

    def __init__(self, commit, diff_data):
        self._commit = commit
        self._diff_data = diff_data

    @property
    def commit(self):
        return self._commit

    @property
    def against(self):
        return None

    def detailed_patches(self):
        return {}

    @cached_property
    def stats(self):

        additions = 0
        deletions = 0

        for f in self._diff_data:
            if f.Additions:
                additions += f.Additions
            if f.Deletions:
                deletions += f.Deletions

        return additions, deletions

    @property
    def changed_files(self):
        changed_files_data = {}
        for f in self._diff_data:
            file_name = f.Path
            extension = self.get_file_extension(file_name)
            changed_files_data[file_name] = {
                'additions': f.Additions or 0,
                'deletions': f.Deletions or 0,
                'extension': extension,
            }
        return changed_files_data

    @property
    def patches(self):
        return []

    def get_patch_data(self):
        return ''
