# coding: utf-8


import logging
import urllib.parse
from ylog.context import log_context

from django.conf import settings
from django.db import OperationalError
from celery.exceptions import SoftTimeLimitExceeded

from intranet.dogma.dogma.core.logic.users import EmailGuesser
from intranet.dogma.dogma.core.logic.commits import get_tracker_data, get_commit_object_for_gerrit

from .git import Repo as GitRepo
from ..crawlers import get_crawler
from ..models import PushedCommit

log = logging.getLogger(__name__)


class Repo(GitRepo):
    def __init__(self, *args, **kwargs):
        super(Repo, self).__init__(*args, **kwargs)
        self.crawler = get_crawler(self.repo.source)
        self.guesser = EmailGuesser()

    def base_clone_command(self, *args, **kwargs):
        self.base_command()

    def base_command(self):
        last_commit = PushedCommit.objects.filter(repo=self.repo.id).order_by('commit_time').last()
        from_date = None
        if last_commit:
            from_date = last_commit.commit_time
        return self.fetch_new_commits(from_date)

    def base_fetch_command(self, credential=None):
        self.base_command()
        return False

    def fetch_new_commits(self, from_date):
        with log_context(repo_id=self.repo.id, repo=str(self.repo)):
            existed_commits = set(PushedCommit.objects.filter(repo=self.repo).values_list('commit', flat=True))
            repo_path = self.repo.vcs_name
            url = urllib.parse.urljoin(self.repo.source.web_url, '/changes/')
            base_params = {
                'n': settings.GERRIT_IN_BATCH,
                'q': 'status:merged project:{}'.format(repo_path),
            }
            if from_date:
                base_params['q'] += ' after:"{}"'.format(from_date.strftime('%Y-%m-%d %H:%M:%S'))
            to_skip = 0
            has_new = False
            while True:
                base_params['S'] = to_skip
                batch = self.crawler.fetch_batch(url=url, params=base_params)
                if not batch:
                    if has_new:
                        self.repo.update_commits_statistics()
                    return
                has_new = True
                self.create_new_commits(batch, existed_commits)
                to_skip += settings.GERRIT_IN_BATCH

    def get_author_data(self, batch):
        authors_map = {}
        for commit in batch:
            for user_id in (commit['owner']['_account_id'], commit['submitter']['_account_id']):
                if user_id not in authors_map:
                    authors_map[user_id] = self.crawler.get_author_by_id(user_id)

        return authors_map

    def create_new_commits(self, batch, existed_commits):
        tracker_data = get_tracker_data(batch)
        authors_data = self.get_author_data(batch)
        commits_to_create = []
        for commit in batch:
            try:
                commit_obj = get_commit_object_for_gerrit(
                    commit=commit, repo=self.repo,
                    authors_data=authors_data, guesser=self.guesser,
                    tracker_data=tracker_data,
                )
                if commit_obj.commit not in existed_commits:
                    commits_to_create.append(commit_obj)
            except (SoftTimeLimitExceeded, OperationalError):
                raise
            except Exception as exc:
                log.exception('Failed to create commit "%s": %s', commit['change_id'], repr(exc))
        PushedCommit.objects.bulk_create(commits_to_create)

    def commits_in_default_branch(self):
        return 0

    def get_commit_native_id(self, repo, commit):
        return commit.commit_id
