# coding: utf-8


import os
import logging
import re
import time
import sh
from django.conf import settings

from .base import BaseRepo
from intranet.dogma.dogma.core.errors.utils import get_parsed_error
from intranet.dogma.dogma.core.errors import BaseError, PermissionError, NoRepositoryError
from intranet.dogma.dogma.core.git.models import Repository
from intranet.dogma.dogma.core.hg.models import to_unicode


log = logging.getLogger(__name__)


class Repo(BaseRepo):
    CHECK_NEW = True

    def get_url(self, credential=None):
        """
        Возвращает url репозитория

        Функция должна быть использована только при вызове команд
        вида git clone, hg clone и тому подобных

        :rtype: basestring
        """
        protocol = self.repo.source.vcs_protocol
        repo_url = None

        if protocol == 'native':
            repo_url = 'git://%s/%s' % (self.repo.source.host, self.repo.vcs_name)
        elif protocol in ('https', 'http'):
            repo_url = '%s://%s/%s' % (protocol, self.repo.source.host, self.repo.vcs_name)
        elif protocol == 'ssh':
            repo_url = 'ssh://%s@%s/%s' % (settings.DOGMA_FETCH_USERNAME, self.repo.source.host, self.repo.vcs_name)

        if repo_url:
            if self.repo.source.web_type == 'gitlab':
                repo_url += '.git'
            return repo_url

        return super(Repo, self).get_url()

    def command_with_credential(self, command, *args, **kwargs):
        credentials = self.repo.get_credentials(auth_type=self.AUTH_TYPE)
        if credentials:
            for credential in credentials:
                try:
                    result = command(credential, *args, **kwargs)
                except (PermissionError, NoRepositoryError) as exc:
                    credential.mark_fail()
                    last_exc = exc
                    continue
                except BaseError as exc:
                    last_exc = exc
                    continue
                else:
                    credential.mark_success()
                    return result
            else:
                raise last_exc
        else:
            return command()

    def clone(self, *args, **kwargs):
        if settings.IS_BUSINESS:
            self.command_with_credential(self.base_clone_command, *args, **kwargs)
        else:
            self.base_clone_command(*args, **kwargs)

    def fetch(self):
        if settings.IS_BUSINESS:
            return self.command_with_credential(self.base_fetch_command)

        return self.base_fetch_command()

    def base_clone_command(self, credential=None):
        self.run_git_command('clone', '--bare', self.get_url(credential), self.path, _in=settings.DOGMA_FETCH_PASSWORD)
        # http://stackoverflow.com/questions/10696718/git-fetch-fails-to-work-on-bare-repo-but-git-pull-works-on-normal-repo
        self.run_git_command('config', 'remote.origin.fetch', '+refs/heads/*:refs/heads/*')

    def run_git_command(self, *args, **kwargs):
        """
        @rtype: sh.RunningCommand
        """
        new_env = os.environ.copy()
        new_env['HOME'] = settings.DOGMA_ROOT
        for key in new_env.keys():
            new_env[key] = str(new_env[key])
        if '_cwd' not in kwargs:
            if os.path.exists(self.path):
                kwargs['_cwd'] = self.path
        try:
            p = sh.git(*args, _env=new_env, **kwargs)

            p.wait()
            return p

        except sh.ErrorReturnCode as error:
            parsed_error = get_parsed_error(error)
            raise parsed_error(error)

    def _remove_lock_if_exists(self):
        try:
            sh.rm('-rf', 'config.lock', _cwd=self.path).wait()
        except sh.ErrorReturnCode as error:
            parsed_error = get_parsed_error(error)
            raise parsed_error(error)

    def prune_clone(self):
        log.info('Clone needs pruning')
        p = sh.rm(os.path.join(self.path, '.git', 'gc.log'), _cwd=self.path)
        p.wait()
        self.run_git_command('prune')

    def has_new_commits(self, result):
        """
        Принимает на вход результат
        выполнения fetch комманды
        возвращает True, если были получены
        новые коммиты, False, если новых
        коммитов нет
        :rtype: bool
        """
        if not self.CHECK_NEW:
            return True
        command_stderr = result.process.stderr
        try:
            command_stderr = to_unicode(command_stderr)
        except Exception as exc:
            log.error('Got error while decoding "%s"', repr(exc))
            return True
        command_stderr = [line for line in command_stderr.split('\n') if '->' in line]
        for branch in command_stderr:
            if not re.search(r'= \[(up to date|актуально)\] {6}', branch):
                return True
        return False

    def base_fetch_command(self, credential=None):

        for i in range(5):
            try:
                self._remove_lock_if_exists()
                self.run_git_command('remote', 'set-url', 'origin', self.get_url(credential))
                result = self.run_git_command('fetch', 'origin', '-v', _in=settings.DOGMA_FETCH_PASSWORD)
                return self.has_new_commits(result)
            except os.error as e:
                if "out of pty devices" in str(e):
                    log.info('Got out of pty error, sleeping')
                    time.sleep(20)
                    continue
                raise
            except BaseError as e:
                base_e = e.args[0]

                # Нередка ошибка на огромных репозиториях
                # Она заканчивается на
                # "warning: Имеется слишком много объектов, на которые нет ссылок;
                # запустите «git prune» для их удаления."
                try:
                    std_err = base_e.stderr
                except AttributeError:
                    std_err = base_e
                if isinstance(std_err, str):
                    if re.search(r'git prune|git remote prune', std_err, re.MULTILINE):
                        self.prune_clone()
                        continue
                    if 'The requested URL returned error: 429' in std_err:
                        log.warning('Got 429 while fetching "%s"', self.repo_id)
                        return False

                raise e

        else:
            raise ValueError('retries ended, sorry')

    def commits_in_default_branch(self):
        """
        git rev-list --count <revision>
        """
        git_repo = Repository.discover(self.path)
        if git_repo.is_empty:
            return 0
        try:
            count = self.run_git_command('rev-list', 'HEAD', '--count')
        except BaseError as exc:
            if '''fatal: bad flag '--count' used after filename''' in str(exc):
                # no default branch
                return 0
            raise
        return int(count.strip() or 0)
