# coding: utf-8


import sh
import re
import time

from django.conf import settings
from mercurial import commands

from intranet.dogma.dogma.core.errors import BaseError, NoRepositoryError
from .base import BaseRepo
from intranet.dogma.dogma.core.utils import get_repository_model


class Repo(BaseRepo):
    def get_url(self, credential=None):
        """
        Поддерживается только https, поэтому fetch работает полагая, что в
        админке никто не изменит урл и протокола по которому происходит хождение
        в hg.
        """
        protocol = self.repo.source.vcs_protocol

        if protocol in ('https', 'http'):
            return '%s://%s:%s@%s/%s' % (protocol,
                                         settings.DOGMA_FETCH_USERNAME,
                                         settings.DOGMA_FETCH_PASSWORD,
                                         self.repo.source.host,
                                         self.repo.vcs_name)
        if protocol == 'ssh':
            return 'ssh://%s@%s//%s' % (settings.DOGMA_FETCH_USERNAME,
                                        self.repo.source.host,
                                        self.repo.vcs_name,
                                        )

        return super(Repo, self).get_url()

    def base_clone_command(self, clone_branches=False, url_pattern=None, credential=None):
        self.run_hg_command('clone', '--uncompressed', self.get_url(credential), self.path)

    def has_new_commits(self):
        repo_model = get_repository_model(self.repo.clones.on_current_node().first())
        repo = repo_model._initial
        repo.ui.setconfig('ui', 'quiet', True)
        result = commands.incoming(repo.ui, repo, source=self.get_url(),  bundle=None, force=False)
        return not result

    def base_fetch_command(self, credential=None):
        fetch_url = self.get_url(credential)
        if self.repo.source.vcs_protocol == 'ssh':
            params = (
                'pull', '-u',
                '--ssh', 'ssh -o StrictHostKeyChecking=no -i {}'.format(
                    settings.DOGMA_SSH_KEY_PATH
                ),
                fetch_url,
            )
            has_new_commits = True
        else:
            params = ('pull', '-u', fetch_url,)
            has_new_commits = self.has_new_commits()
        if has_new_commits:
            for _ in range(5):
                try:
                    self.run_hg_command(*params, _cwd=self.path)
                except BaseError as error:
                    if "out of pty devices" in str(error):
                        time.sleep(20)
                        continue
                    if "run 'hg recover' to clean up transaction" in str(error):
                        self.recover()
                        continue
                    raise
                else:
                    break

        return has_new_commits

    def run_hg_command(self, *args, **kwargs):
        try:
            p = sh.hg(*args, **kwargs)
            p.wait()
            return str(p)
        except sh.ErrorReturnCode as error:
            if re.search(r'remote: abort: repository .*? not found', str(error)):
                raise NoRepositoryError(error)
            raise BaseError(error)

    def recover(self):
        self.run_hg_command('recover', _cwd=self.path)

    def commits_in_default_branch(self):
        """
        hg log --template "\n"

        :return:
        """
        commit_list = self.run_hg_command('log', '--template', '\n', _cwd=self.path)
        return commit_list.count('\n')

    def get_commit_native_id(self, repo, commit):
        extra = commit._initial.extra()
        if 'convert_revision' in extra and 'arcadia' in extra['convert_revision']:
            # 'svn:41d65440-b5be-11dd-afe3-b2e846d9b4f8/trunk/arcadia@2405881'
            return extra['convert_revision'].split('@')[-1]
        return super(Repo, self).get_commit_native_id(repo, commit)
