# coding: utf-8


import re
import logging
import time
import sh

from django.conf import settings

from intranet.dogma.dogma.core.errors import BaseError, EmptyRepositoryError
from .git import Repo as GitRepo


log = logging.getLogger(__name__)


class Repo(GitRepo):
    def get_url(self, credential=None):
        protocol = self.repo.source.vcs_protocol

        if protocol == 'ssh':
            return 'svn+ssh://%s/%s' % (self.repo.source.host, self.repo.vcs_name)
        elif protocol in ('https', 'http'):
            return '%s://%s/%s' % (protocol, self.repo.source.host, self.repo.vcs_name)

        return super(Repo, self).get_url()

    def base_clone_command(self, clone_branches=False, credential=None):
        if not credential:
            self.run_git_command('svn', 'init', '--stdlayout',
                                            '--username', settings.DOGMA_FETCH_USERNAME,
                                            self.get_url(), self.path,
                                            _in=settings.DOGMA_FETCH_PASSWORD)
        else:
            self.run_git_command('svn', 'init', '--stdlayout',
                                 self.get_url(credential), self.path,
                                 )
        self.run_git_command(
            'config', '--local', 'svn.authorsfile', settings.DOGMA_SVN_USERS_PATH,
        )
        self.run_git_command(
            'config', '--local', '--bool', 'core.bare', 'true',
        )
        self.run_git_command(
            'config', '--local', 'svn-remote.svn.fetch', 'trunk:refs/remotes/origin/trunk'
        )
        self.run_git_command(
            'config', '--local', 'svn-remote.svn.branches', 'branches/*:refs/remotes/origin/*'
        )
        self.run_git_command(
            'config', '--local', '--unset', 'svn-remote.svn.tags',
            _ok_code=[0, 5]  # 5 возвращает, когда ключа нет в файле
        )
        if not clone_branches:
            self.run_git_command(
                'config', '--local', '--unset', 'svn-remote.svn.branches',
                _ok_code=[0, 5]
            )

        self.fetch()

    def has_new_commits(self, result):
        command_stdout = result.process.stdout
        return bool(command_stdout)

    def base_fetch_command(self, credential=None):
        for i in range(5):
            try:
                result = self.run_git_command('svn', 'fetch', '-q',
                                              '--authors-prog', '/usr/bin/dogma-userinfo',
                                              '--username', settings.DOGMA_FETCH_USERNAME,
                                              _in=settings.DOGMA_FETCH_PASSWORD)
                return self.has_new_commits(result)
            except BaseError as e:
                base_e = e.args[0]

                try:
                    std_err = base_e.stderr
                except AttributeError:
                    std_err = base_e

                # В git-svn есть какой-то сранный баг, который надо ретраить:
                # http://comments.gmane.org/gmane.comp.version-control.git/258921
                # Воспроизводится на 611 коммите репозитория svn.yandex.ru:bar
                if isinstance(std_err, str):
                    if "out of pty devices" in std_err:
                        time.sleep(20)
                        continue
                    if re.search(r'^write \.git/Git_svn_hash_.*: Bad file descriptor$', std_err[-100:], re.MULTILINE):
                        continue
                    lock_exists = re.search(r'Unable to create \'(?P<path>.+)\': File exists', std_err)
                    if lock_exists:
                        path = lock_exists.group('path')
                        sh.rm(path)
                        continue

                # Нередка ошибка на огромных репозиториях
                # Она заканчивается на
                # "warning: Имеется слишком много объектов, на которые нет ссылок;
                # запустите «git prune» для их удаления."

                    if re.search(r'git prune|git remote prune', std_err, re.MULTILINE):
                        self.prune_clone()
                        continue
                raise e

            else:
                break
        else:
            raise ValueError('retries ended, sorry')

    def get_commit_native_id(self, repo, commit):
        id_ = [commit.hex]

        def repl(match):
            id_[0] = match.groups()[0]

            return ''

        commit.message = re.sub(r'^\s*git-svn-id:.*@(\d*).*$', repl, commit.message, flags=re.S | re.M).strip()

        return id_[0]

    def commits_in_default_branch(self):
        try:
            return super(Repo, self).commits_in_default_branch()
        except BaseError as exc:
            if 'unknown revision or path not in the working tree' in str(exc):
                raise EmptyRepositoryError
            raise
