# coding: utf-8



import collections
import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from requests.exceptions import HTTPError
from ylog.context import log_context

from django.conf import settings
from django.utils.functional import cached_property
from github3.exceptions import AuthenticationFailed, ForbiddenError

from intranet.dogma.dogma.core.models import Source
from intranet.dogma.dogma.core.logic.credential import CredentialClass

log = logging.getLogger(__name__)


BaseRepoTuple = collections.namedtuple('Repo', (
    'name',
    'vcs_name',
    'owner',
    'description',
    'default_branch',
    'is_public',
    'connect_organization',
    'organisation',
))


class RepoTuple(BaseRepoTuple):
    __slots__ = ()

    def __new__(cls, name, vcs_name, owner, description,
                default_branch, is_public,  connect_organization=None, organisation=None
                ):
        if connect_organization is None:
            connect_organization = tuple()
        return super(RepoTuple, cls).__new__(cls, name, vcs_name, owner, description,
                                             default_branch, is_public,  connect_organization,
                                             organisation,
                                             )


class RepoCrawler(CredentialClass):
    def __init__(self, source):
        self.source = source
        self.session = requests.Session()
        retries = Retry(3, status_forcelist={'401'}, backoff_factor=1)
        self.session.mount('http', HTTPAdapter(max_retries=retries))
        self.session.verify = settings.YANDEX_ROOT_CERTIFICATE

    def get_auth(self, credential=None):
        auth = {}
        if credential:
                auth_data = self.get_credential_data(credential, None)
                if auth_data:
                    auth['auth'] = auth_data
                return auth

        if self.source.web_auth == Source.WEB_AUTH_TYPES.basic:
            auth['auth'] = (settings.DOGMA_FETCH_USERNAME, settings.DOGMA_FETCH_PASSWORD)
        elif self.source.web_auth == Source.WEB_AUTH_TYPES.x_oauth_token:
            auth['auth'] = ("x-oauth-token", settings.DOGMA_OAUTH_TOKEN)
        elif self.source.web_auth == Source.WEB_AUTH_TYPES.cookie:
            auth['cookies'] = {'Session_id': settings.DOGMA_SESSIONID}
        elif self.source.web_auth == Source.WEB_AUTH_TYPES.token:
            auth['headers'] = {'Authorization': 'OAuth %s' % settings.DOGMA_OAUTH_TOKEN}

        return auth

    def normalize_name(self, name):
        import future.backports.urllib.parse as urllib_parse
        # XXX see https://bugs.python.org/issue1712522
        return str(urllib_parse.quote(name.replace('/', '-'), safe=''))

    @cached_property
    def _owners_to_copy(self):
        return (
            self.source.organisationstoclone_set
            .filter(is_active=True)
            .prefetch_related('connect_organization')
        )

    def get_repos(self, **kwargs):
        """
        @rtype: RepoTuple
        """
        raise NotImplementedError

    def get_repo_url(self, repo):
        """
        Вернуть URL по которому человек увидит проект в браузере.
        """
        raise NotImplementedError

    def get_commit_url(self, repo, id_):
        """
        Вернуть URL по которому человек увидит комит в браузере.
        """
        raise NotImplementedError

    def repos_by_organisations(self):
        for organisation in self._owners_to_copy:
            with log_context(organisation=str(organisation)):
                credentials = None
                connect_organization = organisation.connect_organization.values_list('id', flat=True)
                if settings.IS_BUSINESS:
                    credentials = organisation.get_credentials(auth_type=self.AUTH_TYPE)
                    if credentials:
                        for credential in credentials:
                            try:
                                for repo in self.repos_by_organisation(
                                        organisation=organisation,
                                        connect_organization=connect_organization,
                                        credential=credential,
                                ):
                                    yield repo
                            except HTTPError as exc:
                                last_exc = exc
                                if exc.response.status_code == 403:
                                        credential.mark_fail()
                                else:
                                    raise
                            except (AuthenticationFailed, ForbiddenError) as exc:
                                last_exc = exc
                                if credential:
                                    log.exception('Not valid credential was provided')
                                    credential.mark_fail()
                            else:
                                credential.mark_success()
                                break
                        else:
                            raise last_exc

                if not settings.IS_BUSINESS or not credentials:
                    for repo in self.repos_by_organisation(
                        organisation=organisation,
                        connect_organization=connect_organization,
                    ):
                        yield repo
