# coding: utf-8

import logging
import urllib.parse

from django.conf import settings

from intranet.dogma.dogma.core.logic.tasks_processing import decode_gerrit_response
from .base import RepoTuple, RepoCrawler


log = logging.getLogger(__name__)


class _GerritUser(object):
    def __init__(self, user_data):
        self.email = user_data['email']
        self.name = user_data['name']
        self.login = user_data['username']


class GerritCrawler(RepoCrawler):
    def get_repos(self, **kwargs):
        url = urllib.parse.urljoin(self.source.web_url, '/projects/')
        base_params = {
            'n': settings.GERRIT_IN_BATCH,
        }
        to_skip = 0
        while True:
            base_params['S'] = to_skip
            batch = self.fetch_batch(url=url, params=base_params)
            if not batch:
                return
            for repo_name in batch.keys():
                try:
                    owner, name = repo_name.split('/', 1)
                except ValueError:
                    name = repo_name
                    owner = 'gerrit'
                yield RepoTuple(
                    name=name,
                    owner=owner,
                    vcs_name=repo_name,
                    description='',
                    default_branch='master',
                    is_public=True,
                    organisation=None,
                )
            to_skip += settings.GERRIT_IN_BATCH

    def fetch_batch(self, url, params=None):
        response = self.session.get(url=url, params=params)
        response.raise_for_status()
        return decode_gerrit_response(response)

    def get_author_by_id(self, author_id):
        url = urllib.parse.urljoin(
            self.source.web_url,
            '/accounts/{}/detail/'.format(author_id),
        )
        return _GerritUser(self.fetch_batch(url=url))

    def get_repo_url(self, repo):
        """
        Вернуть URL по которому человек увидит проект в браузере.
        """
        return '/'.join((self.source.web_url, '/admin/repos/', repo.vcs_name, ))

    def get_commit_url(self, repo, id_):
        """
        Вернуть URL по которому человек увидит комит в браузере.
        """
        return '/'.join((self.source.web_url, 'c', repo.vcs_name, '+', id_, ))
