# coding: utf-8

import itertools
import time
import logging

import github3
from github3.exceptions import GitHubError, NotFoundError
from django.conf import settings
from django.utils.encoding import force_text

from .base import RepoCrawler, RepoTuple

log = logging.getLogger(__name__)


class GithubCrawler(RepoCrawler):

    def api_wrapper(self, credential=None):
        if 'github.com' in self.source.web_url:
            token = settings.DOGMA_GITHUB_EXT_TOKEN
            if credential is not None:
                token = self.get_credential_data(credential, token)
            return github3.GitHub(token=token)
        else:
            return github3.GitHubEnterprise(
                url=self.source.web_url,
                token=settings.DOGMA_GITHUB_INT_TOKEN,
                verify=False
            )

    @property
    def github_repositories(self):
        if self._owners_to_copy:
            return self.repos_by_organisations()

        if settings.IS_BUSINESS:
            return tuple()

        return itertools.chain(
            self.api_wrapper().repositories(type='private'),
            self.api_wrapper().all_repositories()
        )

    def get_repos(self, request_interval=0.05, **kwargs):
        """
        Возвращает информацию о репозиториях

        Параметр request_interval нужен, чтобы
        не создавать сильную нагрузку на github,
        равен интервалу между запросами,
        значение по умолчанию - 50мс

        :type request_interval: int
        """
        for repo in self.github_repositories:
            default_branch = None
            if hasattr(repo, 'default_branch'):
                default_branch = repo.default_branch

            if not default_branch:
                try:
                    br_names = [br.name for br in repo.branches()]
                    if (not br_names) or ('master' in br_names):
                        default_branch = 'master'
                    else:
                        default_branch = sorted(br_names)[0]
                except GitHubError:
                    default_branch = 'master'
                    log.error('Got error response from github')

            organisation = getattr(repo, 'organisation', None)
            connect_organization = getattr(repo, 'connect_organization', None)

            repo = RepoTuple(
                name=force_text(self.normalize_name(repo.name)),
                owner=force_text(repo.owner.login),
                description=force_text(repo.description or ''),
                vcs_name='%s/%s' % (repo.owner.login, repo.name),
                default_branch=force_text(default_branch),
                is_public=not repo.private,
                organisation=organisation,
                connect_organization=connect_organization,
            )
            yield repo
            if request_interval is not None:
                time.sleep(request_interval)

    def repos_by_organisation(self, organisation, connect_organization, credential=None):
        try:
            for repo in self.api_wrapper(credential).repositories_by(organisation.name):
                repo.organisation = organisation
                repo.connect_organization = connect_organization
                yield repo
        except NotFoundError:
            return

    def get_repo_url(self, repo):
        return '%s/%s' % (self.source.web_url, repo.vcs_name)

    def get_commit_url(self, repo, id_):
        return self.get_repo_url(repo) + '/commit/%s' % id_
