# coding: utf-8



import logging
from time import sleep

from cached_property import cached_property
from django.core.exceptions import ImproperlyConfigured
from django.utils.encoding import force_text

from intranet.dogma.dogma.core.dao import connect_organization
from intranet.dogma.dogma.core.models import Source
from .base import RepoCrawler, RepoTuple


log = logging.getLogger(__name__)


class MakesGapError(Exception):
    pass


class GitlabCrawler(RepoCrawler):
    BREAK_AT_GAP = 200

    def _get_token(self):
        if self.source.extra_info.get('token'):
            return self.source.extra_info['token']
        raise ImproperlyConfigured(
            'Go to admin and add {"token": "token"} to Source\'s extra_info'
        )

    def _get_organization(self):
        if self.source.extra_info.get('organization_dir_id'):
            dir_id = int(self.source.extra_info['organization_dir_id'])
            return [connect_organization.get_connect_organization(dir_id)]

    @cached_property
    def _api_version(self):
        return self.source.extra_info.get('api_version') or 'v3'

    def get_auth(self):
        """
        Об аутентификации в gitlab:
        https://github.com/gitlabhq/gitlabhq/tree/master/doc/api#private-tokens
        """
        auth = super(GitlabCrawler, self).get_auth()
        if self.source.web_auth == Source.WEB_AUTH_TYPES.token:
            auth['headers'] = {'PRIVATE-TOKEN': self._get_token()}

        return auth

    @cached_property
    def _auth(self):
        return self.get_auth()

    def get_repos(self, **kwargs):
        """
        Вычитать все репозитории.

        Считается, что если есть пропуск в BREAK_AT_GAP репозиториев между id, то
        дальше проектов нет. Так сделано, потому что ручка
        /api/v3/projects/all доступна только администратору, а просить
        админские права на каждый клонируемый Source не получится.
        """

        log.info('Started to parse gitlab "%s"', self.source)

        # столько последовательных репозиториев допустимо пропустить
        # т.е. еще считаем, что если BREAK_AT_GAP == 1000
        # 1, 2, 3, ... [999 раз получили 404] ... и тут еще может существовать репозиторий
        # но считаем, что
        # 1, 2, 3, ... [1000 раз получили 404] ... и тут пусто
        current_gap = 0
        seen = set()

        index = 0
        while True:
            if current_gap == self.BREAK_AT_GAP:
                log.info(
                    'Finished parsing gitlab at "%s" with gap of "%s"',
                    index, current_gap
                )
                break

            index += 1

            try:
                seen.add(index)
                yield self._try_and_get_repo(index)
            except MakesGapError:
                # поймали пропуск, переходим к следующей записи
                current_gap += 1
                # gitlab у Adfox (git.adfox.ru) иначе не работает
                sleep(0.5)
            else:
                current_gap = 0  # запись есть, поэтому обнуляем длину пропуска
                sleep(0.1)

        try:
            ids = self._mine_projects_ids()
        except Exception as e:
            log.exception("Failed to mine project ids: %s", e)
            return

        ids = sorted(set(ids) - set(seen))
        if ids:
            log.debug("Extra repos found: %s", ids)
        else:
            log.debug("No extra repos found")

        for index in ids:
            yield self._try_and_get_repo(index)

    def _mine_projects_ids(self):
        response = self.session.get(
            self.source.web_url + '/api/{api_verison}/projects'.format(
                api_verison=self._api_version,
            ),
            timeout=60,
            **self._auth
        )

        response.raise_for_status()

        data = response.json()
        return [x["id"] for x in data]

    def _try_and_get_repo(self, repo_id):
        response = self.session.get(
            self.source.web_url + '/api/{api_verison}/projects/{id}'.format(
                api_verison=self._api_version,
                id=repo_id,
            ),
            timeout=5, **self._auth
        )
        if response.status_code in [404, 410]:
            raise MakesGapError

        response.raise_for_status()

        repo = response.json()

        log.info(
            'Gitlab parser got "%s", internal id is "%s"',
            repo['name'], repo_id
        )

        if repo.get('default_branch') is None:
            log.error(
                'Repository "%s" in "%s" has no default branch',
                repo['name'], self.source
            )
            raise MakesGapError

        connect_organization = self._get_organization()
        log.info("Generating github repo with org: %s", connect_organization)

        return RepoTuple(
            name=self.normalize_name(force_text(repo['name']).lower()),
            owner=self.normalize_name(repo['namespace']['path'].lower()),
            vcs_name='{0}'.format(repo['path_with_namespace']),
            description=repo['description'] or '',
            default_branch=force_text(repo['default_branch']),
            is_public=self._repo_is_public(repo),
            connect_organization=connect_organization,
        )

    def _repo_is_public(self, repo):
        status = repo.get('public')
        if status is not None:
            return status
        if repo.get('visibility') == 'private':
            return False
        return True

    def get_repo_url(self, repo):
        return '{0}/{1}'.format(self.source.web_url, repo.vcs_name)

    def get_commit_url(self, repo, id_):
        return '{0}/commit/{1}'.format(self.get_repo_url(repo), id_)
