# coding: utf-8



import logging
import urllib.parse

from django.conf import settings
from django.utils.encoding import force_text

from requests.exceptions import HTTPError

from .base import RepoTuple, RepoCrawler


log = logging.getLogger(__name__)


class StashCrawler(RepoCrawler):
    AUTH_TYPE = 'app_password'

    def get_repos(self, url=None, start=None, **kwargs):
        if self._owners_to_copy:
            repos = self.repos_by_organisations()
        else:
            if settings.IS_BUSINESS:
                return

            repos = self._fetch_all()

        for repo in repos:
            yield repo

    def repos_by_organisation(self, organisation, connect_organization, credential=None):
        for repo in self._fetch_repos(
                organisation.name, credential=credential,
                organisation=organisation, connect_organization=connect_organization,
        ):
            yield repo

    def _fetch_all(self):
        if not self.source.extra_info or self.source.extra_info.get('crawl_projects', True):
            for data in self._fetch_projects():
                yield data
        if not self.source.extra_info or self.source.extra_info.get('crawl_users', True):
            for data in self._fetch_users():
                yield data

    def _get_url(self, url):
        auth = self.get_auth()
        response = self.session.get(url=url, **auth)
        response.raise_for_status()
        return response.json()

    def _make_url(self, type, start, path='/rest/api/1.0/',):
        url = urllib.parse.urljoin(self.source.web_url, path + type)
        if start is not None:
            url += '?start=%s' % start
        return url

    def _fetch_projects(self, start=None):
        data = self._get_url(self._make_url('projects', start))

        for project in data['values']:
            for repo in self._fetch_repos(project['key']):
                yield repo

        if not data['isLastPage']:
            for repo in self._fetch_projects(data['nextPageStart']):
                yield repo

    def _fetch_users(self, start=None):
        data = self._get_url(self._make_url('users', start))

        for project in data['values']:
            for repo in self._fetch_repos('~' + project['name']):  # ~login
                yield repo

        if not data['isLastPage']:
            for repo in self._fetch_users(data['nextPageStart']):
                yield repo

    def get_request_data(self, project_key, credential, next_url):
        project_key = project_key.lower()
        auth = self.get_auth(credential=credential)
        if next_url:
            url = next_url
        else:
            url = urllib.parse.urljoin(self.source.web_url, '/rest/api/1.0/projects/%s/repos' % project_key)
        return url, auth

    def _fetch_repos(self, project_key, credential=None, start=None,
                     organisation=None, connect_organization=None,
                     next_url=None
                     ):

        url, params = self.get_request_data(project_key, credential, next_url)

        if start is not None:
            url += '?start=%s' % start

        response = self.session.get(url=url, timeout=5, **params)
        try:
            response.raise_for_status()
        except HTTPError as exc:
            if exc.response.status_code == 401:
                if credential:
                    credential.mark_fail()
                log.info('Project "%s" does not allow indexing', project_key)
                return
            else:
                raise

        data = response.json()

        for repo in data['values']:
            if settings.IS_BUSINESS:
                default_branch = 'master'
                vcs_name = repo['full_name']
                owner = vcs_name.split('/')[0]
                public = not repo['is_private']
            else:
                project_key = project_key.lower()
                default_branch_url = urllib.parse.urljoin(
                    self.source.web_url,
                    '/rest/api/1.0/projects/%s/repos/%s/branches/default' % (project_key, repo['name']
                                                                             )
                )

                default_branch_response = self.session.get(url=default_branch_url, timeout=5, **params)
                status_code = default_branch_response.status_code
                if status_code == 404 or status_code == 204:
                    continue
                else:
                    default_branch_response.raise_for_status()

                default_branch = default_branch_response.json()['displayId']
                vcs_name = '{}/{}'.format(project_key, '-'.join(repo['name'].split(' ')))
                owner = project_key
                public = self.is_repo_public(project_key, repo)

            yield RepoTuple(
                name=force_text(self.normalize_name(repo['name'])),
                owner=owner,
                vcs_name='{}.git'.format(vcs_name),
                description='',
                default_branch=default_branch,
                is_public=public,
                organisation=organisation,
                connect_organization=connect_organization,
            )
        additional_params = {}
        if settings.IS_BUSINESS:
            next_url = data.get('next')
            if next_url:
                additional_params = {
                    'next_url': next_url,
                }
        else:
            if not data['isLastPage']:
                additional_params = {
                    'start': data['nextPageStart'],
                }

        if additional_params:
            base_params = {
                'credential': credential,
                'organisation': organisation,
                'connect_organization': connect_organization,
            }
            base_params.update(additional_params)
            for repo in self._fetch_repos(project_key, **base_params):
                    yield repo

    def is_repo_public(self, project, repo):
        return repo['public']

    def get_repo_url(self, repo):
        return '%s/projects/%s/repos/%s/browse' % (repo.source.web_url, repo.owner, repo.name)

    def get_commit_url(self, repo, id_):
        return '%s/projects/%s/repos/%s/commits/%s' % (repo.source.web_url, repo.owner, repo.name, id_)
