# coding: utf-8

import logging
import itertools
import datetime
from datetime import timedelta
from dateutil.relativedelta import relativedelta

from celery.exceptions import SoftTimeLimitExceeded

from dir_data_sync.org_ctx import get_org

from django.db import OperationalError
from django.db import connection
from django.db import transaction
from django.conf import settings

from intranet.dogma.dogma.core.models import Clone, PushedCommit, Node, PushedCommitDuplicate, PushedCommitDuplicateHex
from intranet.dogma.dogma.core.dao.users import guess_user
from intranet.dogma.dogma.core.dao.repo import get_repo
from intranet.dogma.dogma.core.logic.commits import (get_commit_message, get_commit_diff,
                                      get_parents_objects, get_commit_type,
                                      )
from intranet.dogma.dogma.core.dao.users import get_users_by
from intranet.dogma.dogma.core.dao.changed_file import create_changed_files

log = logging.getLogger(__name__)


def commits_by_source(source):
    return PushedCommit.objects.filter(repo__source=source)


def get_commits_by_sha(sha, source, owner, name, since):
    repo = get_repo(name, owner, source)
    commits = PushedCommit.objects.filter(repo=repo)
    if sha:
        commits = commits.filter(branch_name=sha)
    if since:
        commits = commits.filter(commit_time__gte=since)
    return commits.prefetch_related()


def get_commits_diff_by_batch(batch):
    commits_hexs = (commit.hex for commit in batch)
    return PushedCommit.objects.filter(commit__in=commits_hexs).values('lines_added', 'lines_deleted', 'commit')


def get_commits_map(repo):
    return {
            pushed_commit.commit: pushed_commit
            for pushed_commit in PushedCommit.objects.filter(repo=repo)
        }


def get_duplicate_commits_map(repo):
    return {
            pushed_commit.commit: pushed_commit
            for pushed_commit in PushedCommitDuplicate.objects.filter(repo=repo)
        }


def update_pushed_commit_data(pushed_commit, author, committer, additions,
                              deletions, message, branch, tree_hex, commit_id, parents_objects):
    pushed_commit.author_id = author.id
    pushed_commit.committer_id = committer.id
    pushed_commit.lines_added = additions
    pushed_commit.lines_deleted = deletions
    pushed_commit.message = message
    pushed_commit.branch_name = branch
    pushed_commit.tree_hex = tree_hex
    pushed_commit.commit_id = commit_id
    pushed_commit.save()
    pushed_commit.parents.add(*parents_objects)


def get_commits_by_query(include_query, exclude_query):
    commits = (PushedCommit.objects
               .filter(**include_query)
               .exclude(**exclude_query)
               .select_related('author',
                               'committer',
                               'repo',
                               'repo__source',
                               )
               )
    if settings.IS_BUSINESS:
        commits = commits.filter(repo__connect_organization=get_org())
    return commits


def create_commit_from_data(commit_data, parents_objects):
    """
    Создаем коммит на основе переданных данных,
    ранее проверили что коммитов с такими хешеми нет.
    """
    model = PushedCommit

    commit = model.objects.create(**commit_data)
    if parents_objects:
        commit.parents.add(*parents_objects)

    return commit


def get_all_repo_commits(clone, repo_model):
    """
    Вернуть все комиты всего репозитория.
    """
    return repo_model.all_commits(exclude=get_known_commits_by_repo(clone.repo))


def get_known_commits_by_repo(repo):
    return set(
        itertools.chain(
            PushedCommit.objects.filter(repo__name=repo.name, repo__source=repo.source).values_list('commit', flat=True,),
            PushedCommitDuplicateHex.objects.filter(repo=repo).values_list('commit', flat=True,),
        )
    )


def filter_duplicate_commits(commits, repository):
    duplicates = set(PushedCommit.objects.filter(commit__in=[commit.hex for commit in commits]).values_list('commit', flat=True, ))
    unique_commits = [commit for commit in commits if commit.hex not in duplicates]
    PushedCommitDuplicateHex.objects.bulk_create([PushedCommitDuplicateHex(repo=repository, commit=duplicate) for duplicate in duplicates])
    return unique_commits


def create_commits_objects(commits, commit_map, diff_data, repo_raw,
                           guesser, repository, tracker_data,
                           changed_files_map, pushed_commits=None
                           ):
    """
    Получаем информацию по коммитам с файловой системы и
    записыаем ее в базу
    """
    for commit in commits:
        try:
            commit_diff = diff_data.get(commit.hex)
            message = get_commit_message(commit.message, commit_diff)
            commit_id = repository.backend.get_commit_native_id(repo_raw, commit)
            author = guess_user(commit.author, guesser, message)
            committer = guess_user(commit.committer, guesser, message)
            branch = getattr(commit, "branch_name", repository.default_branch)
            parents = [parent.hex for parent in commit.parents]
            additions, deletions = get_commit_diff(commit_diff)
            parents_objects = get_parents_objects(commit_map, parents)
            queues, tickets = tracker_data[commit.hex]
            commit_type = get_commit_type(commit)

            commit_data = {
                'commit': commit.hex,
                'repo': repository,
                'commit_time': commit.commit_time,
                'queues': queues,
                'commit_type': commit_type,
                'tickets': tickets,
                'author_id': author.id,
                'committer_id': committer.id,
                'lines_added': additions,
                'lines_deleted': deletions,
                'message': message,
                'branch_name': branch,
                'tree_hex': commit.tree.hex,
                'commit_id': commit_id,
            }
            pushed_commit = create_commit_from_data(commit_data, parents_objects,)
            commit_map[commit.hex] = pushed_commit

            if pushed_commits is not None:
                pushed_commits.append(pushed_commit)

            if len(parents) != len(parents_objects):
                log.warning('Found fewer parents than expected for commit "%s", repo "%s"',
                            commit.hex,
                            repository.name,
                            )
            changed_files_data = changed_files_map.get(commit.hex)
            if changed_files_data:
                log.info('Found changed files, creating')
                create_changed_files(changed_files_data, pushed_commit)
                log.info('Successfully create changed files')
                # TODO Убрать код ниже после создания данных о файлах всех существующих коммитов
                pushed_commit.create_changed_files = True
            else:
                log.info('Found no changed files for "%s"', commit.hex)
                pushed_commit.create_changed_files = False
            pushed_commit.save()

        except (SoftTimeLimitExceeded, OperationalError):
            raise
        except Exception as exc:
            log.exception('Failed to create commit "%s": %s', commit.hex, repr(exc))

    return pushed_commits


@transaction.atomic
def update_commits_count(source):
    total = PushedCommit.objects.filter(repo__source=source).count()
    source.total_commits = total
    source.save()


def update_commits_stats_for_repo(repo, ):
    """
    @type repo: Repo
    """
    total_commits = PushedCommit.objects.filter(repo=repo).count()
    repo.known_commits = total_commits
    if total_commits:
        last_commit = PushedCommit.objects.filter(repo=repo).order_by('-commit_time').first()
        if last_commit:
            repo.contiguous_chain_of_commits_ends_at_id = last_commit.id


def bind_commits_to_user(new_user, *old_users):
    """
    Привязываем все коммиты переданных пользователей
    к заданному
    """

    for commit_model in (PushedCommit, PushedCommitDuplicate):
        commit_model.objects.filter(author__in=old_users).update(author=new_user)
        commit_model.objects.filter(committer__in=old_users).update(committer=new_user)


def attach_commits_to_uid_by_emails(uid, *bad_emails):
    """
    Привязываем все коммиты пользователей с указанными
    емейлами к заданному пользователю
    """
    log.info('Started bind commit process')
    with transaction.atomic():
        users = get_users_by('email', *bad_emails)
        user_to_bind = get_users_by('uid', uid).first()
        bind_commits_to_user(user_to_bind, *users)
        log.info('Successfully bind commits to "%s"', uid)
        users.delete()
        log.info('Successfully delete users: "%s"', bad_emails)


def get_commits_for_file_statistics_aggregation(date, exact_only=False):
    query = {
        'aggregated__isnull': True,
        'create_changed_files': True,
        'commit_type': PushedCommit.TYPES.common,
    }
    if exact_only:
        query.update({
            'commit_time__range': (date,
                                   date + timedelta(hours=23, minutes=59, seconds=59),
                                   )
        })
    else:
        query.update({'commit_time__gte': date.date()})
    return PushedCommit.objects.filter(**query)


def get_commits_stat_by_users_by_months(logins):
    start_from = datetime.datetime.now() - relativedelta(months=6)
    start_from = datetime.datetime(start_from.year, start_from.month, 1, 0, 0, 0, 0)

    with connection.cursor() as c:
        c.execute("""
            SELECT
                login,
                COUNT(commit.id) as commits_count,
                SUM(commit.lines_added) as lines_added,
                SUM(commit.lines_deleted) as lines_deleted,
                EXTRACT(MONTH FROM commit.commit_time) AS month,
                EXTRACT(YEAR FROM commit.commit_time) AS year
            FROM core_user
            LEFT JOIN core_pushedcommit AS commit
            ON commit.author_id = core_user.id
            WHERE
                core_user.login in (%s) AND
                commit.commit_time > '%s'
            GROUP BY month, year, login
            ORDER BY year DESC, month DESC
        """ % (
            ", ".join("'{}'".format(l) for l in logins),
            start_from,
        ))
        return c.fetchall()


def get_commits_count_by_day(login, start, end, offset):
    with connection.cursor() as c:
        c.execute("""
            SELECT
                login,
                COUNT(commit.id) as commits_count,
                DATE(commit.commit_time + INTERVAL '%s seconds') as day
            FROM core_user
            LEFT JOIN core_pushedcommit AS commit
            ON commit.author_id = core_user.id
            WHERE
                commit.commit_time >= '%s'
                AND commit.commit_time <= '%s'
                AND login = '%s'
            GROUP BY login, day
        """ % (
            offset.seconds,
            start,
            end,
            login
        ))

        return c.fetchall()
