# coding: utf-8


import operator
import logging
from django.db.models.query_utils import Q
from django.utils.timezone import now, timedelta
from django.conf import settings

from ylog.context import log_context
from dir_data_sync.org_ctx import get_org

from intranet.dogma.dogma.core.models import Repo, Source, Node, Clone
from intranet.dogma.dogma.core.utils import weighted_shuffle
from functools import reduce

log = logging.getLogger(__name__)

DEFAULT_DC = ('man', 'sas', 'myt')

SOURCES_MAP = {
        'github': ('myt', 'sas',
                   ),
        'nocgitlab': ('man', 'sas',
                      ),
        'browser-stash': ('man', 'myt',
                          ),
        'stash-qe': ('sas', 'myt',
                     ),
        'intgit': ('man', 'myt',
                   ),
        'adfox': ('sas', 'myt',
                  ),
        'hg': ('sas', 'man',
               ),
        'svn': ('myt', 'sas',
                ),
        'git': ('sas', 'myt',
                ),
    }


def get_repo_by_id(repo_id):
    return Repo.objects.filter(id=repo_id).first()


def get_repo(name, owner, source_code):
    return Repo.objects.filter(name=name, owner=owner, source__code=source_code).first()


def get_straggled_repos(important=None):
    show_to = now() - timedelta(hours=4)
    source_fail_recently = now() - timedelta(hours=6)

    objects = Repo.objects.filter((Q(source__last_sync_fail_time__lt=source_fail_recently) |
                                   Q(source__last_sync_fail_time__isnull=True)),
                                  last_sync_success_time__lt=show_to,
                                  source__status=Source.SYNC_STATUSES.success,
                                  source__hidden=False,
                                  is_active=True,
                                  on_remote=True,)
    if important:
        objects = objects.filter(is_important=True)
    if settings.IS_BUSINESS:
        objects = objects.filter(clones__isnull=False)
    return objects


def get_failed_repos(important=None):
    fail_to = now() - timedelta(hours=4)
    objects = Repo.objects.filter(status=Repo.SYNC_STATUSES.fail,
                                  source__hidden=False,
                                  created__lt=fail_to)
    if important:
        objects = objects.filter(is_important=True)
    return objects


def select_nodes_for_repo(repo):
    """
    Возвращает список нод подходящих для
    клонирования/переноса репозитория,
    учитывает то, что источники
    следует селить в определенных ДЦ

    Возвращает список нод, на которых еще нет клона
    репозитория и которые соответстует местоположению
    источника (если местоположение указано
    в SOURCES_MAP), причем ноды с большим количеством
    свободного места более вероятно окажутся на
    первых позициях в списке

    :param repo: Repo
    :rtype: list(Node)
    """

    nodes = reduce(operator.or_, (Q(hostname__contains=node) for
                                  node in SOURCES_MAP.get(repo.source.code,
                                                          DEFAULT_DC)))

    nodes = Node.objects.filter(nodes,
                                ~Q(id__in=Clone.objects.filter(repo=repo).values_list('node_id',
                                                                                      flat=True)
                                   ),
                                enabled=True,
                                ).order_by('-space_available')
    return weighted_shuffle(nodes, [node.space_available for node in nodes])


def make_repo_local(repo):
    repo.on_remote = False
    repo.status = Repo.SYNC_STATUSES.success
    repo.save()


def get_or_create_repo(vcs_name, name, owner, url, source,):
    created = False
    try:
        repo = Repo.objects.get(source=source, vcs_name=vcs_name)
    except Repo.DoesNotExist:
        repo = Repo.objects.create(
            source=source,
            vcs_name=vcs_name,
            name=name,
            owner=owner,
            url=url,
        )
        created = True
    return repo, created


def disable_repos(*repos):
    """
    Удаляем связь репозиториев с организациями из коннекта
    Если больше связанных организаций из коннекта не осталось -
    перестаем индексировать репозиторий
    """
    repos_ids = [repo.id for repo in repos]
    with log_context(repos=repos_ids):
        log.info('Removing organization from repos')
        RepoOrgModel = Repo.connect_organization.through
        RepoOrgModel.objects.filter(repo_id__in=repos_ids, organization_id=get_org().id).delete()
        log.info('Successfully remove organization from repos')
        Repo.objects.filter(pk__in=repos_ids, connect_organization__isnull=True).update(is_active=False)
