# coding: utf-8



import itertools
import logging

from ylog.context import log_context

from intranet.dogma.dogma.core.models import User

log = logging.getLogger(__name__)


def guess_user(user_data, guesser, message=None):
    user, trust = guesser.guess_user_by(user_data.email, message)
    if not user:
        user = User.objects.create(
            login=user_data.login,
            email=user_data.email,
            name=user_data.name,
            from_staff=False,
        )
    elif not trust and user.from_staff:
        user.from_staff = False
        user.save()

    return user


def guess_main_user(user):
    if user.uid:
        return user
    with log_context(email=str(user.email)):
        if '@' not in user.email:
            return
        main_users = User.objects.filter(
            uid__isnull=False,
            from_staff=True,
            other_emails__regex=r',{}|^{}'.format(user.email, user.email),
        )

        if len(main_users) > 1:
            log.error('Found more than one main user')

        main_user = main_users.first()
        if main_user:
            return main_user
        log.info('Found no main user for')


def get_users_other_emails(users_logins):
    users_other_emails = User.objects.filter(login__in=users_logins,
                                             from_staff=True).values_list('other_emails',
                                                                          flat=True)
    return itertools.chain.from_iterable((emails.split(',') for emails in users_other_emails))


def get_users_id_by_login(users_login):
    other_emails = get_users_other_emails(users_login)
    return list(get_users_by('email', *other_emails).values_list('id', flat=True))


def create_or_update_user_from_data(data, user_for_update=None):
    is_new = False
    if user_for_update is not None:
        user = user_for_update
    else:
        try:
            user = User.objects.get(uid=data['uid'])
        except User.DoesNotExist:
            user = User(uid=data['uid'])
            is_new = True

    name_components = [data['name']['first']['ru'].strip(),
                       data['name']['last']['ru'].strip()]
    other_emails = itertools.chain((email['address'] for email in data['emails']),
                                   (account['value'] for account in data['accounts']
                                    if account['type'] == 'another_work_email'),
                                   [data['work_email']],
                                   )
    attrs = {
        'uid': data['uid'],
        'login': data['login'],
        'email': data['work_email'],
        'name': ' '.join([_f for _f in name_components if _f]),
        'other_emails': ','.join(set(other_emails)),
        'from_staff': True,
    }

    is_changed = False
    for attr, value in attrs.items():
        if is_new or getattr(user, attr) != value:
            is_changed = True
            setattr(user, attr, value)

    if is_new or is_changed:
        user.save()

    return user


def get_users_by(key, *values):
    format_key = '{}__in'.format(key)
    filter_param = {format_key: values}
    return User.objects.filter(**filter_param)
