# -*- coding: utf-8 -*-


import socket
import ssl
import logging
import re
from requests import exceptions as requests_exceptions

from django.utils.translation import ugettext_lazy as _

from . import (
    NetworkError,
    ConnectionError,
    BaseError,
    RemoteServiceError,
    PermissionError,
    EmptyRepositoryError,
    NoRepositoryError,
    NoRepositoryOnNodeError,
    SlowRemoteResponse,
)

from . import ERRORS_MAP

log = logging.getLogger(__name__)


def get_parsed_error(initial_error, trace=None):
    """
    Определяет обобщенный вид ошибки и возвращает его

    Смотрит либо на тип ошибки либо на текст (так как
    большинство действий выполняются через sh, и тип ошибки
    будет всегда sh.ErrorReturnCode, а изначальная ошибка
    будет в текстовом виде в сообщении ошибки
    """
    unicode_error = str(initial_error) + str(trace)
    if isinstance(initial_error, BaseError):
        return initial_error
    elif any(error_text in unicode_error for error_text in ('403 Forbidden',
                                                            'URL access forbidden for unknown reason',
                                                            'Authentication failed for',
                                                            'Invalid username or password',
                                                            )
             ):
        return PermissionError
    elif any(error_text in unicode_error for error_text in ('Read timed out',
                                                            'Connection timed out',
                                                            )
             ):
        return SlowRemoteResponse
    elif 'internal server error' in unicode_error:
        return RemoteServiceError
    elif 'fatal: Couldn\'t find remote ref HEAD' in unicode_error:
        return EmptyRepositoryError
    elif 'requested repository does not exist, or you do not have permission' in unicode_error:
        return NoRepositoryError
    elif re.search(r'fatal: repository .*? not found', unicode_error):
        return NoRepositoryError
    elif any(error_text in unicode_error for error_text in ('OSError: [Errno 2] No such file or directory',
                                                            'error.RepoError(_("repository %s not found")',
                                                            'remote: Not Found',
                                                            )
             ):
        return NoRepositoryOnNodeError
    elif isinstance(initial_error, (socket.error, ssl.SSLError)):
        return NetworkError
    elif isinstance(initial_error, requests_exceptions.ConnectionError):
        return ConnectionError
    elif isinstance(initial_error, requests_exceptions.RequestException):
        return NetworkError
    return BaseError


def get_error_value(db_value):
    error_class = ERRORS_MAP.get(db_value)
    if error_class:
        return error_class.help
    log.error('No error with such code: "%s"', db_value)
    return _('Неизвестный тип ошибки, пожалуйста, обратитесь в поддержку')
