# coding: utf-8
import re
import os
import gc
from datetime import datetime
import pytz
from base64 import b64encode
import logging
import pygit2

from schematics.models import Model
from schematics.types import (StringType, LongType, DateTimeType, BooleanType,
                              SHA1Type, IntType, EmailType)
from schematics.types.compound import ListType, ModelType

from .. import abstract_repository


log = logging.getLogger(__name__)


def dtfromts(ts):
    timestamp = datetime.utcfromtimestamp(ts)
    utc = pytz.timezone('UTC')
    return utc.localize(timestamp)


class Base(Model):
    def _convert(self, raw, **kwargs):
        fields = kwargs.get('fields', list(self._fields.keys()))
        return {name: getattr(raw, name) for name in fields}

    def __init__(self, raw_data, *args, **kwargs):
        self._initial = raw_data
        super().__init__(raw_data, *args, **kwargs)


class Repository(Base):
    path = StringType()
    workdir = StringType()
    is_bare = BooleanType()
    is_empty = BooleanType()

    @classmethod
    def discover(cls, path):
        try:
            path = pygit2.discover_repository(path)
        except KeyError:
            log.error('No repository found at "%s"', path)
            raise
        return cls(pygit2.Repository(path))

    def lookup_note(self, *args, **kwargs):
        return self._initial.lookup_note(*args, **kwargs)

    def list_branches(self, type_='local'):
        types = {
            'local': pygit2.GIT_BRANCH_LOCAL,
            'remote': pygit2.GIT_BRANCH_REMOTE,
            'both': pygit2.GIT_BRANCH_LOCAL | pygit2.GIT_BRANCH_REMOTE,
        }
        assert type_ in types

        return self._initial.listall_branches(types[type_])

    def branch_head(self, branch_name):
        head = self._initial.lookup_branch(branch_name).peel()
        if head is not None:
            return Commit(head)

    def get_branch(self, branch_name):
        br = self._initial.lookup_branch(branch_name)
        if br is not None:
            return Branch(br)

    def get(self, hex_, default=None):
        res = self._initial.get(hex_)

        if res is not None:
            return map_to_git_class(res)
        else:
            return default

    def __getitem__(self, hex_):
        res = self.get(hex_)
        if res is None:
            raise KeyError(hex_)
        return res

    @property
    def head(self):
        return Reference(self._initial.head)

    def get_object_by_refspec(self, refspec):
        try:
            commit = self._initial.revparse_single(refspec)
            if commit.type == pygit2.GIT_OBJ_TAG:
                commit = commit.peel(pygit2.GIT_OBJ_COMMIT)
            return map_to_git_class(commit)
        except KeyError:
            raise ValueError("Ref %s point to nonexistent object" % refspec)

    def get_object_by_path(self, commit, path):
        if path == '':
            return commit.tree

        try:
            tree_entry = commit.tree[path]
        except KeyError:
            raise ValueError("Path %s doesnot exists" % path)
        else:
            if tree_entry.filemode == pygit2.GIT_FILEMODE_COMMIT:
                # значит это submodule
                return Submodule(tree_entry._initial)

            try:
                return self[tree_entry.hex]
            except KeyError:
                log.error("Can't get tree entry %s on path %s, commit %s in repo %s",
                          tree_entry.hex, path, commit.hex, self.path)
                raise

    def get_tree_entries(self, commit, path):
        def rec(res, path, inside=False):
            # если дерево и мы не заглядывали внутрь, то заглядываем
            if isinstance(res, Tree) and not inside:
                return [rec(self[obj.hex], os.path.join(path, obj.name), True)
                        for obj in res if obj.filemode != pygit2.GIT_FILEMODE_COMMIT]

            if isinstance(res, Blob):
                return ('file', path, res)
            elif isinstance(res, Tree):
                return ('dir', path, res)
            elif isinstance(res, Submodule):
                return ('submodule', path, res)
            else:
                raise RuntimeError('Unexpected type %s, waight for Tree or Blob' %
                                   type(res))

        return rec(self.get_object_by_path(commit, path), path)

    def commit_diff(self, commit, against=None):
        """
        Дифф коммита, по умолчанию относительно первого предка
        Если against == -1, то относительно пустого дерева

        """
        repo = self._initial
        commit = commit._initial
        diff = None

        if against is None and commit.parents:
            against = commit.parents[0]
        elif against is None or against == -1:
            # если нет предков, или явно просили дифф с пустым деревом
            diff = commit.tree.diff_to_tree(
                swap=True, context_lines=3)
        else:
            against = against._initial

        if diff is None:
            diff = repo.diff(against, commit, context_lines=3)
            diff.find_similar()

        return Diff(diff)

    def walk(self, hex, sort_type=''):
        """ sort_type:
          'tp' - topological,
          'tm' - time,
          'tp+rv' - topological reverse,
          'tm+rv' - time reverse
        """
        types = {
            '': pygit2.GIT_SORT_NONE,
            'tp': pygit2.GIT_SORT_TOPOLOGICAL,
            'tm': pygit2.GIT_SORT_TIME,
            'tp+rv': pygit2.GIT_SORT_TOPOLOGICAL | pygit2.GIT_SORT_REVERSE,
            'tm+rv': pygit2.GIT_SORT_TIME | pygit2.GIT_SORT_REVERSE,
        }

        return Walker(self._initial.walk(hex, types[sort_type]))

    def commits_between(self, cm1, cm2):
        """ Поиск наименьшего общего предка """
        walker = self._initial.walk(cm2.hex, pygit2.GIT_SORT_TOPOLOGICAL)
        walker.hide(cm1.hex)
        return Walker(walker)

    def all_commits(self, exclude):
        """
        Вернуть все комиты всего репозитория упорядоченные по дате от старых к новым.

        @rtype: list
        """
        seen_commits = set()
        for branch_name in self.list_branches():
            log.info(f'Getting commits for branch "{branch_name}"')
            branch_head = self.branch_head(branch_name)
            if not branch_head:
                log.info(f'No branch head for "{branch_name}"')
                continue
            if branch_head.hex in seen_commits:
                continue
            for native_commit in self._initial.walk(branch_head.hex, pygit2.GIT_SORT_TIME | pygit2.GIT_SORT_REVERSE):
                if native_commit.hex in seen_commits:
                    continue
                if native_commit.hex in exclude:
                    continue
                seen_commits.add(native_commit.hex)
                yield Commit(native_commit)


class Walker(object):
    def __init__(self, initial):
        self._initial = initial

    def __iter__(self):
        for i in self._initial:
            yield Commit(i)

    def hide(self, hex):
        self._initial.hide(hex)


class User(Base):
    login = StringType()
    uid = StringType()
    name = StringType()
    email = EmailType()
    time = DateTimeType()

    def _convert(self, raw, **kwargs):
        info = super()._convert(raw, fields=('name', 'email'), **kwargs)
        info.update({
            'login': raw.email.split('@')[0],
            'time': dtfromts(raw.time),
            'uid': None,
        })
        return info


class Branch(Base):
    branch_name = StringType()
    name = StringType()
    target = SHA1Type()


class GitObject(Base):
    hex = SHA1Type()

    def __hash__(self):
        return hash(self.hex)


class Blob(GitObject):
    data = StringType()
    size = LongType()
    is_binary = BooleanType()

    def encode_data(self):
        try:
            return 'utf-8', self.data.decode('utf-8')
        except UnicodeDecodeError:
            return 'base64', b64encode(self.data)


class Tree(GitObject):
    def __iter__(self):
        for i in self._initial:
            yield TreeEntry(i)

    def __getitem__(self, name):
        return TreeEntry(self._initial[name])


class TreeEntry(GitObject):
    name = StringType()
    filemode = IntType()


class Commit(GitObject):
    author = ModelType(User)
    committer = ModelType(User)
    message = StringType()
    tree = ModelType(Tree)
    parent_ids = ListType(SHA1Type())
    commit_time = DateTimeType()

    @property
    def parents(self):
        return list(map(self.__class__, self._initial.parents))

    @property
    def merge_commit(self):
        return len(self.parents) > 1

    def _convert(self, raw, **kwargs):
        data = super()._convert(
            raw, fields=('hex', 'tree'), **kwargs)

        try:
            data['message'] = raw.message
        except LookupError:  # unknown encoding: ru_RU.KOI8-R
            data['message'] = '! unrecoverable KOI8-R message in repository'
        data['author'] = User(raw.author)
        data['committer'] = User(raw.committer)
        data['parent_ids'] = list(map(str, raw.parent_ids))
        data['tree'] = Tree(raw.tree)
        data['commit_time'] = dtfromts(raw.author.time)

        return data

    def __repr__(self):
        return '<Commit: %s>' % self.hex[:8]


class Reference(Base):
    name = StringType()
    shorthand = StringType()
    type = StringType(choices=('simbolic', 'object'))

    @property
    def target(self):
        if self.type == 'simbolic':
            return self._initial.target
        else:
            return self._initial.target.hex


class Patch(Base):
    line_stats = ListType(IntType)
    additions = IntType()
    deletions = IntType()

    def _convert(self, raw, **kwargs):
        fields = 'line_stats',
        data = super()._convert(raw, fields=fields, **kwargs)
        data['additions'] = raw.line_stats[1]
        data['deletions'] = raw.line_stats[2]
        return data


class Diff(abstract_repository.ChangedFilesMixin, Base):
    patch = StringType()

    # На дифах с большим количеством файлов получение значения patch приводит к OOM
    # Будем помечать такие дифы как "толстые"
    is_fat = BooleanType()

    def _convert(self, raw, **kwargs):
        data = super()._convert(raw, fields=('patch',), **kwargs)
        data['is_fat'] = False
        return data

    def patches(self, patches_objects):
        return list(map(Patch, patches_objects))

    def detailed_patches(self):
        SPLIT_PATCH_TXT_RE = re.compile(
            r'^\+\+\+\ b\/([^\n]*?)\n(@@.*?)(?=\n^diff|\n\Z)', re.M | re.S)

        if self.patch is None:
            return {}

        matches = re.findall(SPLIT_PATCH_TXT_RE, self.patch)
        return dict(m for m in matches)

    def get_patches_objects(self):
        MAX_PATCHES = 200
        patches = []
        for patch in self._initial:
            patches.append(patch)
            if len(patches) == MAX_PATCHES:
                self.is_fat = True
                log.info('Got too many patches objects while getting diff')
                return
        return patches

    @property
    def stats(self):
        patch_objects = self.get_patches_objects()
        if not patch_objects:
            gc.collect()
            return 0, 0
        patches = self.patches(patch_objects)
        additions = 0
        deletions = 0
        for patch in patches:
            additions += patch.additions
            deletions += patch.deletions
        return additions, deletions

    def get_patch_data(self):
        if not self.is_fat:
            return self.patch


class Submodule(GitObject):
    pass


def map_to_git_class(obj):
    MAPPING = {
        pygit2.Blob: Blob,
        pygit2.Tree: Tree,
        pygit2.TreeEntry: TreeEntry,
        pygit2.Commit: Commit,
        pygit2.Reference: Reference,
    }

    return MAPPING[obj.__class__](obj)
