# coding: utf-8

import logging
from os import path

from django.conf import settings
import chardet
from cached_property import cached_property
from schematics.types import StringType, DateTimeType, SHA1Type
from schematics.types.compound import ListType, ModelType
from pretend import stub
from intranet.dogma.dogma.core.git.models import dtfromts

from mercurial import hg, ui, node, context, patch, error, revset, logcmdutil
from .. import abstract_repository

from . import diff_manifests

log = logging.getLogger(__name__)


class Repository(abstract_repository.Repository):
    SORT = {
        '': 'time_desc',  # от позднего к раннему
        'tp': 'time_desc',  # в гите это TOPOLOGICAL
        'tm': 'time_desc',
        'tp+rv': 'time_asc',  # от раннего к позднему
        'tm+rv': 'time_asc',
    }

    @classmethod
    def discover(cls, path):
        user_interface = ui.ui()
        user_interface.setconfig(b'ui', b'report_untrusted', b'off')
        encoded_path = path.encode('utf-8')
        hg_repo = hg.repository(ui=user_interface, path=encoded_path)
        return cls(hg_repo)

    def __init__(self, *args, **kwargs):
        super(Repository, self).__init__(*args, **kwargs)
        hg_repo = self._initial
        self.diff_manifests = diff_manifests.DiffManifests(hg_repo)

    def _convert(self, raw, **kwargs):
        revs = revset.spanset(raw)
        info = super(Repository, self)._convert(raw, fields=('path',), **kwargs)
        info.update({
            'workdir': path.dirname(raw.path),
            'is_bare': False,
            'is_empty': len(revs) == 0,
        })
        return info

    def list_branches(self, type_='local'):
        """
        Возвращает только локальные бранчи!
        """
        types = {'local'}
        assert type_ in types
        return [k for k, _ in self._initial.branchmap().iteritems()]

    def branch_head(self, branch_name):
        heads = self._initial.branchheads(branch_name)
        if heads:
            head = node.hex(heads[0])  # берем условно-произвольный HEAD
            rev_node, rev = diff_manifests.get_node_rev_by_changeid(head, self._initial)
            return Commit(context.changectx(self._initial, rev, rev_node))

    def get_branch(self, branch_name):
        branch_head = self.branch_head(branch_name)
        if branch_head:
            return abstract_repository.Branch(stub(
                branch_name=branch_name,
                name=branch_name,
                target=branch_head.hex))

    def get_object_by_path(self, commit, path):
        log.info('This is not implemented correctly for HG')
        return stub(
            hex='00000000000000',  # this is not implemented
        )

    def get_tree_entries(self, commit, path):
        raise NotImplementedError
        # используется в content-view, readme-view

    def commit_diff(self, commit, against=None):
        """
        Дифф коммита, по умолчанию относительно первого предка
        Если against == -1, то относительно пустого дерева

        """
        commit = commit._initial
        # diff = None

        if against is None and commit.parents():
            against = commit.parents()[0]
        elif against is None or against == -1:
            raise NotImplementedError
            # TODO: сделать
            # если нет предков, или явно просили дифф с пустым деревом
            # diff = commit.tree.diff_to_tree(
            #     swap=True, context_lines=context_lines)
        else:
            against = against._initial

        if settings.FAST_HG_DIFFS:
            # Подменяю манифесты комитов для ускорения расчета дифов
            against_manifest, commit_manifest = self.diff_manifests.get(against, commit)
            against._manifest = against_manifest
            commit._manifest = commit_manifest

        return Diff((commit, against))

    def walk(self, hex, sort_type=''):
        """
        Сортировка реализовано не точь в точь как в гите, но по смыслу близка.
        И на самом деле это может вызвать проблемы на больших репозиториях, где
        топологическая сортировка и сортировка по времени отличаются.

        Кроме того мы берем историю только дефолтного бранча. Если вам стала
        нужна история не-дефолтного бранча, этот код необходимо доработать!

        @param hex: всегда включается в результат обхода

        Если сортировка от позднего к раннему (по-умолчанию) то hex будет первым,
        а последним будет самый ранний коммит.
        Если сортировка от раннего к позднему то hex будет первым, а последним
        будет самый поздний комит.
        """
        assert sort_type in self.SORT
        wopts = logcmdutil.walkopts({}, {}, [])
        raw_revisions = logcmdutil.getrevs(self._initial, wopts)[0]

        def commit_generator():
            seen_hex = False
            for raw_revision in raw_revisions:
                raw_commit = self._initial[raw_revision]
                if raw_commit.hex() == hex:
                    seen_hex = True
                if seen_hex:
                    yield raw_commit

        commits = commit_generator()

        if self.SORT[sort_type] == 'time_asc':
            commits = reversed(list(commits))

        return Walker(commits)

    def commits_between(self, cm1, cm2):
        """ Поиск наименьшего общего предка """
        # тут надо использовать _initial.changelog.nodesbetween?
        raise NotImplementedError

    def lookup_note(self, *args, **kwargs):
        raise NotImplementedError

    def get(self, hex_, default=None):
        rev_node, rev = diff_manifests.get_node_rev_by_changeid(hex_, self._initial)
        changectx = context.changectx(self._initial, rev, rev_node)
        return Commit(changectx)

    def get_object_by_refspec(self, refspec):
        try:
            rev_node, rev = diff_manifests.get_node_rev_by_changeid(refspec, self._initial)
            ctx = context.changectx(self._initial, rev, rev_node)
        except error.RepoLookupError:
            raise ValueError  # признак что нет бранча, смотри CommitListView
        return Commit(ctx)

    @property
    def head(self):
        branch_head = self.branch_head('default')
        if not branch_head:
            raise ValueError  # пустой бранч по-умолчанию
        return Reference(branch_head)

    def all_commits(self, exclude):
        """
        Вернуть комиты основного бранча! Не всех бранчей.

        Упорядоченные по дате от старых к новым.

        @rtype: list
        """
        wopts = logcmdutil.walkopts({}, {}, [])
        raw_revs = logcmdutil.getrevs(self._initial, wopts)[0]
        raw_revs.reverse()
        for raw_revision in raw_revs:
            raw_commit = self._initial[raw_revision]
            if raw_commit.hex() in exclude:
                continue
            yield Commit(raw_commit)


class User(abstract_repository.User):

    def _convert(self, raw, **kwargs):
        """

        @param user: "Vladislav Senin <troy4eg@yandex-team.ru>"
        @type user: tuple
        """
        user, commit_time = raw
        user = user.strip()
        username_email = user.rsplit(' ', 1)
        if len(username_email) < 2:
            login = user
            name = user
            email = user
        else:
            email = username_email[1].strip('<>')
            name = username_email[0].strip()
            login = email.split('@')[0]

        info = {
            'login': login,
            # использований этого времени не нашел,
            # в тестах использовал то что написал
            'time': commit_time,
            'uid': None,
            'email': email,
            'name': name,
        }
        return info


class Tree(abstract_repository.GitObject):
    def _convert(self, raw, **kwargs):
        return {
            'hex': 'not_implemented_for_HG_repository'
        }

    def __iter__(self):
        raise NotImplementedError

    def __getitem__(self, name):
        raise NotImplementedError


class Commit(abstract_repository.GitObject):
    author = ModelType(User)
    committer = ModelType(User)
    message = StringType()
    tree = ModelType(Tree)
    parent_ids = ListType(SHA1Type())
    commit_time = DateTimeType()

    @property
    def parents(self):
        return list(map(self.__class__, self._initial.parents()))

    @property
    def merge_commit(self):
        return len(self.parents) > 1

    def _convert(self, raw, **kwargs):
        # в терминах HG author и commiter это одно и то же
        commit_message = ''
        if raw.description():
            commit_message = to_unicode(raw.description())
        commit_time = dtfromts(raw.date()[0])
        user = User([to_unicode(raw.user()), commit_time])
        data = {
            'hex': raw.hex(),
            'author': user,
            'committer': user,
            'message': commit_message,
            'parent_ids': [str(p.hex()) for p in raw.parents()],
            'commit_time': commit_time,
            'tree': Tree({}),
        }
        return data

    def __repr__(self):
        return '<Commit: %s>' % self.hex[:8]

    __str__ = __repr__


def build_stat_by(diff_head, diff_body):
    diff_head = diff_head.replace('\n', ' ')
    combined = [diff_head]
    combined += diff_body.split('\n')
    encoded_lines = [line.encode('UTF-8') for line in combined]
    filename, adds, removes, is_binary = patch.diffstatdata(encoded_lines)[0]
    return {
        'filename': filename,
        'additions': adds,
        'deletions': removes,
        'is_binary': is_binary,
    }


class Diff(abstract_repository.ChangedFilesMixin, abstract_repository.Diff):
    MAX_DIFF = 500

    @property
    def commit(self):
        return self._initial[0]

    @property
    def against(self):
        return self._initial[1]

    @property
    def _hg_diff(self):
        head = None
        for i, chunk in enumerate(self.commit.diff(self.against)):
            if i % 2:
                body = to_unicode(chunk)
                yield head, body
            else:
                head = to_unicode(chunk)

    def detailed_patches(self):
        result = {}

        for head, body in self._hg_diff:
            filename = head.split('\n')[0].rsplit(' ', 1)[-1].strip()
            result[filename] = body

        return result

    @cached_property
    def stats(self):

        additions = 0
        deletions = 0

        for i, data in enumerate(self._hg_diff):
            if i == self.MAX_DIFF:
                log.info('Got too many changes while getting diff')
                return 0, 0
            head, body = data
            info = build_stat_by(head, body)
            additions += info['additions']
            deletions += info['deletions']

        return additions, deletions

    @property
    def patches(self):
        return [
            Patch([self.commit, self.against, head, body])
            for head, body in self._hg_diff
        ]

    def get_patch_data(self):
        commit_diff = self.commit.diff(self.against)
        result_patch = ''
        for i, commit_patch in enumerate(commit_diff):
            if i == self.MAX_DIFF:
                log.info('Got too many changes while getting data for changed files')
                return ''
            try:
                unicode_patch = to_unicode(commit_patch)
                result_patch += unicode_patch
            except UnicodeDecodeError:
                log.warning('Got unicode error while parsing commit diff')
                return ''
        return result_patch


def to_unicode(bytestring):
    if isinstance(bytestring, str):
        return bytestring

    try:
        return bytestring.decode('utf-8')
    except UnicodeDecodeError:
        try:
            return bytestring.decode('Windows-1251')
        except UnicodeDecodeError:
            try:
                return bytestring.decode('latin-1')
            except UnicodeDecodeError:
                result = chardet.detect(bytestring)
                return bytestring.decode(result['encoding'])


class Patch(abstract_repository.Patch):
    def _convert(self, raw, **kwargs):
        """
        @type raw: tuple
        @param raw: filename, patch
        """
        commit_current, commit_against, diff_head, diff_body = raw
        # diff -r 70c40abbae16 -r 61f159bc24a0 auto-core/src\n
        splitted_patch = diff_head.split('\n')
        # --- a/auto-core/RegionUtils.java   Tue Jun 28 16:02:47 2016 +0300
        previous_filename = splitted_patch[1].split(' ')[1][2:].split('\t')[0]
        # +++ b/auto-core/RegionUtils.java   Thu Jun 30 14:28:11 2016 +0300
        current_filename = splitted_patch[2].split(' ')[1][2:].split('\t')[0]

        status_name = 'modified'
        if previous_filename == '/dev/null':
            status_name = 'added'
        elif current_filename == '/dev/null':
            status_name = 'removed'
        elif previous_filename != current_filename:
            status_name = 'renamed'

        patch_info = build_stat_by(diff_head, diff_body)

        data = {
            'old_file_path': previous_filename,
            'new_file_path': current_filename,
            'old_hex': commit_against.hex(),
            'new_hex': commit_current.hex(),
            'status_name': status_name,
            'status': self.GIT_STATUS_TO_NAME[status_name],
            'additions': patch_info['additions'],
            'deletions': patch_info['deletions'],
            'is_binary': patch_info['is_binary'],
        }

        return data


class Walker(abstract_repository.Walker):
    def __iter__(self):
        for commit in self._initial:
            yield Commit(commit)


class Reference(abstract_repository.Base):
    name = StringType()
    shorthand = StringType()
    type = StringType(choices=('simbolic', 'object'))

    def _convert(self, raw, **kwargs):
        """
        Весьма условно маплю в git-аналог Reference.
        """
        return {
            'name': 'default',
            'shorthand': 'default',
            'type': 'object',
        }

    @property
    def target(self):
        return self._initial.hex
