# coding: utf-8


import time
import logging
import functools

from multiprocessing.pool import ThreadPool
from celery.exceptions import SoftTimeLimitExceeded

from django.core.urlresolvers import reverse
from django.conf import settings
from django.http import HttpRequest

from intranet.dogma.dogma.api.utils import make_absolute_url
from intranet.dogma.dogma.core.logic.repos import get_repo_url
import intranet.dogma.dogma.core.logic as logic
from intranet.dogma.dogma.core.models import BaseCommit, PushedCommit
from intranet.dogma.dogma.core.dao.users import guess_user

from ids.registry import registry
from ids.exceptions import IDSException


formatter = registry.get_repository('formatter', 'formatter',
                                    user_agent='dogma', retries=4,
                                    host=settings.DOGMA_WF_API_HOST,
                                    protocol='https',
                                    )

WF_CONFIG = 'intranet'
if settings.IS_BUSINESS:
    WF_CONFIG = 'biz'
PRIVATE_REPO_MESSAGE = '****** Коммит в приватный репозиторий'

log = logging.getLogger(__name__)


class SourceInfo(object):
    min_commits = 0
    max_commits = 0


def commits_by_sha(repo, sha=None, sort_order='reverse_time'):
    """
    Вернуть комиты из репозитория.

    @rtype: iterable
    @param sort_order: если reverse_time, то сортировать от раннего к позднему
                       иначе сортировать от позднего к раннему
    """
    if sha is None:
        try:
            start = repo.head.target
        except ValueError:  # пустой бранч
            return []  # тут неявно возвращаем пустой список!
    else:
        try:
            start = repo.get_object_by_refspec(sha).hex
        except ValueError:  # no such branch
            return []

    return repo.walk(start, 'tp' if sort_order == 'topological' else 'tm+rv')


def get_commit_diff(commit_diff):
    return commit_diff if commit_diff else (0, 0)


def get_commit_message(message, commit_diff):
    return message if commit_diff else message + ' (diff unknown)'


def get_stats_pattern(additions, deletions):
    return {
                'stats': {
                    'additions': additions,
                    'deletions': deletions,
                    'total': additions + deletions,
                },
                'files': [],
            }


def get_commit_url(request, repo, commit):
    if repo.source.vcs_type == 'gerrit':
        return 'not implemented'
    return make_absolute_url(
        request,
        path=reverse('api:commit', args=[commit.commit])

    )


def get_user_pattern(commit):
    user = commit.author.for_push
    user.update({'date': commit.commit_time})
    return user


def get_push_pattern(commit, detailed='short', request=None, need_parents=True):
    if not request:
        request = logic.commits._build_request()
    repo = commit.repo

    author = get_user_pattern(commit)
    committer = get_user_pattern(commit)

    commit_url = get_commit_url(request, repo, commit)
    html_url = repo.source.crawler.get_commit_url(repo, commit.commit_id)
    if repo.is_public or settings.IS_BUSINESS:
        commit_message = commit.message
    else:
        commit_message = PRIVATE_REPO_MESSAGE

    data = {
        'url': commit_url,
        'sha': commit.commit,
        'repo': {
            'name': repo.vcs_name,
            'url': get_repo_url(repo)
        },
        '_dogma': {
            'id': commit.commit_id,
            'html_url': html_url,
        },
        'commit': {
            'url': commit_url,
            'author': author,
            'committer': committer,
            'message': commit_message,
            'queues': commit.queues,
            'tickets': commit.tickets,
            'tree': {
                'url': make_absolute_url(
                        request,
                        path='not_implemented_for_hg',
                    ),
                'sha': commit.tree_hex,
            },
        },
        'branch_name': [commit.branch_name],
        'parents': get_parents_pattern(commit, request, repo, need_parents),
    }
    if detailed != 'short':
        data.update(get_stats_pattern(additions=commit.lines_added,
                                      deletions=commit.lines_deleted))
    return data


def get_parents_pattern(commit, request, repo, need_parents):
    parents = []
    if need_parents:
        parents = [
            {
                'url': get_commit_url(request, repo, parent),
                'sha': parent.commit,
            } for parent in commit.parents.all()
        ]
    return parents


def get_parents_objects(commit_map, parents):
    return [commit_map[parent] for parent in parents
            if commit_map.get(parent)]


def get_tickets_queues_for_commit(commit, request_interval=None):
    message = getattr(commit, 'message', None)
    if not message:
        message = commit.get('subject')
    queues = tickets = None

    if message:
        try:
            wiki_node = formatter.get_structure(message, config=WF_CONFIG, version=5,)
        except IDSException as exc:
            log.error('Got error from formatter "%s", for: "%s"', repr(exc), repr(commit.message))
        except Exception as exc:
            log.error('Got error "%s", for: "%s"', repr(exc), repr(commit.message))
        else:
            queues = set()
            tickets = set()
            for ticket_node in wiki_node.filter(type='jira'):
                ticket = ticket_node['wiki-attrs']
                queue = ticket['project']
                ticket_key = '{}-{}'.format(queue, ticket['ticket'])
                queues.add(queue)
                tickets.add(ticket_key)
            queues = list(queues)
            tickets = list(tickets)

        if request_interval is not None:
            time.sleep(request_interval)
    return commit, queues, tickets


def get_commit_type(commit):
    return BaseCommit.TYPES.merge if commit.merge_commit else BaseCommit.TYPES.common


def _build_request():
    request = HttpRequest()
    request.META = {
        'SERVER_NAME': settings.DOGMA_HOST,
        'SERVER_PORT': 443,
    }
    return request


def slice_commits_in_batches(commit_generator, slice_size=None):
    """
    Формируем пачки коммитов для добавления в базу
    возвращаем результат если размер списка
    коммитов достиг SLICE_SIZE или если больше нет коммитов,
    не добавленных в базу
    """
    SLICE_SIZE = slice_size or 100
    commits = list()
    while True:
        try:
            commit = next(commit_generator)
        except StopIteration:
            if commits:
                yield commits
            return

        commits.append(commit)

        if len(commits) == SLICE_SIZE:
            yield commits
            commits = list()


def get_diff_data_for_batch(batch_diff, diff_map):
    """
    Операция медленная, не следует
    запускать ее в транзакции
    """
    for commit_hex, diff in batch_diff.items():
        if commit_hex in diff_map:
            continue
        try:
            additions, deletions = diff.stats
            diff_map[commit_hex] = (additions, deletions)
        except SoftTimeLimitExceeded:
            raise
        except Exception as exc:
            log.exception('Got exception while getting commit diff "%s"', repr(exc))
    return diff_map


def get_tracker_data(batch):
    pool = ThreadPool(5)
    try:
        results = pool.imap_unordered(functools.partial(get_tickets_queues_for_commit,
                                                                 request_interval=0.05),
                                               batch)

        tracker_data = {}
        for commit, queues, tickets in results:
            hex = getattr(commit, 'hex', None)
            if not hex:
                hex = commit['change_id']
            tracker_data[hex] = (queues, tickets)
        return tracker_data
    finally:
        pool.close()
        pool.join()


def get_diff_map(diff_data):
    diff_map = {}
    for commit_diff in diff_data:
        commit = commit_diff['commit']
        lines_added = commit_diff['lines_added']
        lines_deleted = commit_diff['lines_deleted']
        diff_map[commit] = (lines_added, lines_deleted)
    return diff_map


def get_batch_diff(batch, repo_raw):
    return {
        commit.hex: repo_raw.commit_diff(commit)
        for commit in batch
    }


def get_commit_object_for_gerrit(commit, repo, authors_data, guesser, tracker_data):
    author_id = commit['owner']['_account_id']
    committer_id = commit['submitter']['_account_id']
    commit_id = commit['_number']
    author = guess_user(authors_data[author_id], guesser)
    committer = guess_user(authors_data[committer_id], guesser)
    branch = repo.default_branch
    additions, deletions = commit['insertions'], commit['deletions']
    message = commit['subject']
    queues, tickets = tracker_data.get(commit['change_id'], (None, None))
    commit_type = BaseCommit.TYPES.common

    commit_data = {
        'commit': '{}_{}'.format(commit['change_id'], commit_id),
        'repo': repo,
        'commit_time': commit['submitted'],
        'queues': queues,
        'commit_type': commit_type,
        'tickets': tickets,
        'author_id': author.id,
        'committer_id': committer.id,
        'lines_added': additions,
        'lines_deleted': deletions,
        'message': message,
        'branch_name': branch,
        'tree_hex': '',
        'commit_id': commit_id,
        'create_changed_files': True,
    }
    return PushedCommit(**commit_data)
