import logging
import json

from django.utils import timezone
from django.db import OperationalError
from django import db
from django.conf import settings

from psycopg2 import InterfaceError

MAX_SYNC_DELAY = 24 * 7
MIN_SYNC_DELAY = 1

log = logging.getLogger(__name__)


def mark_object_as_failed(model_object, trace, parsed_error):
    """
    Used in case if task failed and we need to store
    an information about error
    :type model_object: models.Model
    :type trace: string
    :type parsed_error: Exception
    """
    trace = trace[-4000:]  # write only last 4000 symbols
    model_object.last_sync_fail_trace = trace
    model_object.last_sync_fail_time = timezone.now()
    model_object.status = model_object.SYNC_STATUSES.fail
    model_object.last_sync_fail_error_code = parsed_error.db_value
    if hasattr(model_object, 'sync_delay'):
        new_delay = model_object.sync_delay * 2
        if new_delay > MAX_SYNC_DELAY:
            new_delay = MAX_SYNC_DELAY
        model_object.sync_delay = new_delay

    log.info('Mark object "%s" as failed', model_object)


def mark_object_as_successful(model_object):
    """
    Used in case if task succeed
    :type model_object: models.Model
    """
    model_object.last_sync_fail_trace = None
    model_object.last_sync_fail_error_code = None
    model_object.last_sync_success_time = timezone.now()
    model_object.status = model_object.SYNC_STATUSES.success
    if hasattr(model_object, 'sync_delay'):
        model_object.sync_delay = MIN_SYNC_DELAY
    log.info('Mark object "%s" as successful', model_object)


def retry_operational_error(task, *args, **kwargs):
    for _ in range(2):
        try:
            return task(*args, **kwargs)
        except (OperationalError, InterfaceError) as exc:
            db.connections.close_all()
            log.error('Got db error "%s"', repr(exc))
            continue


def decode_gerrit_response(response):
    """
    Strip off Gerrit's magic prefix and decode a response.
    :returns:
        Decoded JSON content as a dict, or raw text if content could not be
        decoded as JSON.
    """
    default_response = {}
    content_type = response.headers.get('content-type', '')
    content = response.content.strip()
    if response.encoding:
        content = content.decode(response.encoding)
    if not content:
        return default_response
    if content_type.split(';')[0] != 'application/json':
        return default_response
    if content.startswith(settings.GERRIT_JSON_PREFIX):
        content = content[len(settings.GERRIT_JSON_PREFIX):]
    try:
        return json.loads(content)
    except ValueError:
        log.error('Invalid json content')
        raise
